import { setToQueryValue } from '@yandex-infracloud-ui/libs-next';
import { IHostFilters } from './api_models/hosts';

export type MetaStatusValue = 'all' | 'dead' | 'error' | 'processing' | 'queue';

export interface IMetaStatusFilterButton {
   id: MetaStatusValue;
   name: string;
}

export interface IMetaStatusValues {
   all: number;
   dead: number;
   error: number;
   processing: number;
   queue: number;
}

export class MetaStatusButtons {
   public static readonly config: IMetaStatusFilterButton[] = [
      { id: 'all', name: 'All' },
      { id: 'dead', name: 'Dead' },
      { id: 'error', name: 'With errors' },
      { id: 'processing', name: 'Processing' },
      { id: 'queue', name: 'Queue' },
   ];

   /**
    * Утверждено:
    *    all:        status = ''             health = ''
    *    dead:       status = 'dead,invalid' health = ''
    *    error:      status = 'error'        health = ''
    *    processing: status = 'task'         health = ''
    *    queue:      status = 'steady'       health = 'failure'
    */
   public static patchFilters(filters: IHostFilters, value: MetaStatusValue): IHostFilters {
      const newFilters: IHostFilters = { ...filters };

      switch (value) {
         case 'all':
            newFilters.status = new Set();
            newFilters.health = new Set();
            break;

         case 'dead':
            newFilters.status = new Set(['dead', 'invalid']);
            newFilters.health = new Set();
            break;

         case 'error':
            newFilters.status = new Set(['error']);
            newFilters.health = new Set();
            break;

         case 'processing':
            newFilters.status = new Set(['task']);
            newFilters.health = new Set();
            break;

         case 'queue':
            newFilters.status = new Set(['steady']);
            newFilters.health = new Set(['failure']);
            break;

         default:
            console.warn(`Unknown MetaStatus value: ${value}`);
            break;
      }

      return newFilters;
   }

   public static getValueFromFilters(filters: IHostFilters): MetaStatusValue {
      const status = setToQueryValue(filters.status);
      const health = setToQueryValue(filters.health);

      if (status === 'dead,invalid' && health === '') {
         return 'dead';
      }

      if (status === 'error' && health === '') {
         return 'error';
      }

      if (status === 'task' && health === '') {
         return 'processing';
      }

      if (status === 'steady' && health === 'failure') {
         return 'queue';
      }

      return 'all';
   }

   public percents: IMetaStatusValues = {
      all: 1,
      dead: 0,
      error: 0,
      processing: 0,
      queue: 0,
   };

   constructor(public values: IMetaStatusValues = { all: 0, dead: 0, error: 0, processing: 0, queue: 0 }) {
      this._recalculatePercents();
   }

   public update(values: Partial<IMetaStatusValues>) {
      return new MetaStatusButtons({ ...this.values, ...values });
   }

   private _recalculatePercents() {
      this.percents = {
         all: 1,
         dead: this.values.dead / this.values.all,
         error: this.values.error / this.values.all,
         processing: this.values.processing / this.values.all,
         queue: this.values.queue / this.values.all,
      };
   }
}
