import { IValidationResult, validateEntity } from '@yandex-infracloud-ui/libs-next';
import { config } from 'services';

import { addValidationMatchers } from '../../utils/forTests';

import { IAutomationPlot, plotSchema, plotShape, plotShapeWithOwners, plotSchemaWithOwners } from './automation_plots';

/**
 * Эти тесты - попытка разобраться в библиотеке для валидации yup.
 */
describe('automation_plots models', () => {
   describe('isPlotValid', () => {
      addValidationMatchers();
      config.installationType = 'internal';

      function isPlotValid(plot: IAutomationPlot): IValidationResult<IAutomationPlot> {
         const fields = Object.keys(config.isExternal ? plotShape : plotShapeWithOwners) as (keyof IAutomationPlot)[];

         return validateEntity(config.isExternal ? plotSchema : plotSchemaWithOwners, fields, plot as any, null); // TODO typings
      }

      it('should failed without id', () => {
         const plot: IAutomationPlot = {
            checks: [],
            id: '',
            name: 'Plot name',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeInvalid('id', ['ID is a required field']);
      });

      it('should failed with too long id', () => {
         const plot: IAutomationPlot = {
            id: 'plot_id_0123456789_0123456789_0123456789',
            name: 'Plot name',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeInvalid('id', ['ID must be at most 32 characters']);
      });

      it('should check id with regexp', () => {
         const create = (id: string) =>
            ({
               id,
               name: 'Plot name',
               owners: new Set(['khoden']),
            } as IAutomationPlot);

         expect(isPlotValid(create('wall-e-hw-checks'))).toBeValid();

         expect(isPlotValid(create('a-abc-xyz'))).toBeValid();

         expect(isPlotValid(create('plot id'))).toBeInvalid('id', ['ID should consist of chars from [a-z0-9-]']);
      });

      it('should failed without name', () => {
         const plot: IAutomationPlot = {
            id: 'plot-id',
            name: '',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeInvalid('name', ['Name is a required field']);
      });

      it('should failed with too long name', () => {
         const plot: IAutomationPlot = {
            id: 'plot-id',
            name: 'Very vary vary vary vary long long long long long long name',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeInvalid('name', ['Name must be at most 32 characters']);
      });

      it('should failed without owners', () => {
         const plot: IAutomationPlot = {
            id: 'plot-id',
            name: 'Plot name',
            owners: new Set([]),
         };

         expect(isPlotValid(plot)).toBeInvalid('owners', ['At least one owner should be added']);
      });

      it('should failed with invalid owners', () => {
         const plot: IAutomationPlot = {
            id: 'plot-id',
            name: 'Plot name',
            owners: new Set(['Артём']),
         };

         expect(isPlotValid(plot)).toBeInvalid('owners', ['Owner should consist of chars from [a-z0-9_-@]']);
      });

      it('should success', () => {
         const plot: IAutomationPlot = {
            checks: [],
            id: 'plot-id',
            name: 'Plot name',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeValid();
      });

      it('should failed if contains invalid check', () => {
         const plot: IAutomationPlot = {
            checks: [
               {
                  enabled: true,
                  name: 'Некорректное значение',
                  reboot: true,
                  redeploy: true,
               },
            ],
            id: 'plot-id',
            name: 'Plot name',
            owners: new Set(['khoden']),
         };

         expect(isPlotValid(plot)).toBeInvalid('checks', ['Check name should consist of chars from [a-zA-Z0-9-.]']);
      });
   });
});
