import { setAsArray } from '@yandex-infracloud-ui/libs-next';
import { array, boolean, object, Schema, string } from 'yup';

import { OWNER_REGEXP } from '../regexps';

export const allAutomationPlotFields = ['id', 'name', 'owners', 'checks', 'yc_iam_folder_id'];

/**
 * Check properties
 */
export interface ICheck {
   /**
    * Enable check
    */
   enabled?: boolean;

   /**
    * Juggler check name
    *
    * "maxLength": 32,
    * "pattern": "^[a-zA-Z][a-zA-Z0-9_.]*(?:-[a-zA-Z0-9_.]+)*$",
    */
   name: string;

   /**
    * Try to reboot host to fix failure
    */
   reboot?: boolean;

   /**
    * Try to redeploy host to fix failure
    */
   redeploy?: boolean;

   profile?: boolean;

   /**
    * Try to report about problem
    */
   report_failure?: boolean;

   /**
    * Try to wait until failure fixes itself
    */
   wait?: boolean;

   start_time?: number;
}

export interface IAutomationPlot {
   checks?: ICheck[];

   /**
    * A unique ID
    *
    * "maxLength": 32,
    * "pattern": "(^[a-z][a-z0-9]*(?:-[a-z0-9]+)*$|wall-e-hw-checks)",
    */
   id: string;

   /**
    * Name that will identify automation plot in UI
    *
    * "maxLength": 32,
    * "minLength": 1,
    */
   name: string;

   /**
    * Automation plot owners
    *
    * "pattern": "^([a-zA-Z0-9_][a-zA-Z0-9_-]{,38}[a-zA-Z0-9_]|@[a-zA-Z0-9_][a-zA-Z0-9_-]{,100}[a-zA-Z0-9_])$",
    */
   owners?: Set<string>;

   yc_iam_folder_id?: string;
}

export interface IPlotActionParams {
   /**
    * An optional reason string
    */
   reason?: string;
}

const plotCheckSchema = object().shape({
   enabled: boolean().label('Enabled').required(),

   name: string()
      .label('Check name')
      .required()
      .max(32)
      .matches(/^[a-z][\w.]*(?:-[\w.]+)*$/i, 'Check name should consist of chars from [a-zA-Z0-9-.]'),

   reboot: boolean().label('Reboot').required(),

   redeploy: boolean().label('Redeploy').required(),
});

export const plotShape = {
   id: string()
      .label('ID')
      .required()
      .max(32)
      .matches(/^[a-z][a-z\d]*(?:-[a-z\d]+)*$/, 'ID should consist of chars from [a-z0-9-]'),

   checks: array().label('Checks').of(plotCheckSchema),

   name: string().label('Name').required().max(32),
};

export const plotShapeWithOwners = {
   ...plotShape,

   owners: (setAsArray()
      .label('Owners')
      .required('At least one owner should be added')
      .of(string().matches(OWNER_REGEXP, 'Owner should consist of chars from [a-z0-9_-@]')) as any) as Schema<
      Set<string>
   >, // TODO
};

export const plotSchema = object().shape(plotShape);
export const plotSchemaWithOwners = object().shape(plotShapeWithOwners);
