import { NullableDate } from '@yandex-infracloud-ui/libs-next';
import {
   CmsTaskAction,
   DeployNetwork,
   HostHealthStatus,
   MacAddress,
   OperationState,
   OwnershipValue,
   Provisioner,
} from '../enums';
import { IDeployConfig } from './new_project';

export const allHostFields = [
   'active_mac',
   'active_mac_source',
   'active_mac_time',
   'config',
   'deploy_config_policy',
   'deploy_network',
   'deploy_tags',
   'extra_vlans',
   'health.check_statuses',
   'health.decision',
   'health.event_time',
   'health.message',
   'health.reasons',
   'health.status',
   'infiniband_info.cluster_tag',
   'infiniband_info.ports',
   'inv',
   'ipmi_mac',
   'ips',
   'juggler_aggregate_name',
   'location.city',
   'location.country',
   'location.datacenter',
   'location.network_source',
   'location.network_timestamp',
   'location.physical_timestamp',
   'location.port',
   'location.queue',
   'location.rack',
   'location.short_datacenter_name',
   'location.short_queue_name',
   'location.switch',
   'macs',
   'messages',
   'name',
   'operation_state',
   'platform',
   'project',
   'provisioner',
   'restrictions',
   'scenario_id',
   'state',
   'state_audit_log_id',
   'state_author',
   'state_expire',
   'state_reason',
   'state_time',
   'state_timeout',
   'status',
   'status_audit_log_id',
   'status_author',
   'status_reason',
   'status_time',
   'status_timeout',
   'task.audit_log_id',
   'task.error',
   'task.owner',
   'task.status',
   'task.status_message',
   'ticket',
   'tier',
   'type',
   'uuid',
];

/**
 * Для вывода в списке хостов
 */
export const shortHostFields = [
   'health.check_statuses',
   'health.message',
   'inv',
   'location.city',
   'location.port',
   'location.switch',
   'messages',
   'name',
   'project',
   'restrictions',
   'scenario_id',
   'state',
   'state_reason',
   'status',
   'status_reason',
   'task.status',
   'task.status_message',
   'ticket',
   'tier',
   'type',
   'uuid',
];

/**
 * Состояние хоста
 *
 * По идее грузится в /constants, но для удобства сделал в виде перечисления
 */
export enum HostState {
   Assigned = 'assigned',
   Maintenance = 'maintenance',
   Free = 'free',
}

export interface IParsedQuery {
   invs: number[];
   names: string[];
   patterns: string[];
   tags: string[];
   uuids: string[];
}

export interface IHostHealthStatuses {
   [key: string]: HostHealthStatus;
}

/**
 * Contains info about current host health status
 *
 * ok < broken < failure
 */
export enum HealthStatus {
   Broken = 'broken',
   Failure = 'failure',
   Ok = 'ok',
}

export interface IHostHealth {
   check_statuses: IHostHealthStatuses;
   message?: string;
   reasons?: string[];
   status: HealthStatus;
}

export interface IHostHealthRecord {
   fqdn: string;
   id: string;
   metadata: any;
   status: HostHealthStatus;
   status_mtime: number | Date;
   timestamp: number | Date;
   type: string;
}

export interface IHostFilters {
   config: Set<string>;
   fqdn: string;
   health: Set<string>;
   ids?: Set<number>;
   locations: Set<string>;
   ownership: OwnershipValue;
   project: Set<string>;
   restrictions: Set<string>;
   state: Set<string>;
   status: Set<string>;
   switch: Set<string>;
   type: Set<string>;
}

export type SortDirection = 'asc' | 'desc' | null;
export type SortName =
   | 'links' // scenario_id && ticket
   | 'inv'
   | 'name'
   | 'project'
   | 'state'
   | 'status'
   | 'tier'
   | 'switch'
   | 'health'
   | 'datacenter'
   | 'type';

export interface IHostSort {
   name: SortName;
   direction: SortDirection;
}

export interface IHostLocation {
   city: string;
   country: string;
   datacenter: string;
   network_source: string;
   network_timestamp: number | Date;
   physical_timestamp: number | Date;
   port: string;
   queue: string;
   rack: string;
   short_datacenter_name: string;
   short_queue_name: string;
   switch: string;
}

export enum HostMessageSeverity {
   Error = 'error',
   Info = 'info',
}

export interface IHostMessage {
   message: string;
   severity: HostMessageSeverity;
   source?: string;
}

export interface IHostMessages {
   [key: string]: IHostMessage[];
}

export interface IHostStateExpire {
   issuer: string;
   status: string;
   ticket: string;
   time?: number;
}

export interface IHost {
   active_mac: MacAddress;
   active_mac_source: string;
   active_mac_time: number | Date;
   config: string;
   deploy_config_policy?: string;
   deploy_network?: DeployNetwork;
   deploy_tags?: Set<string>;
   health: IHostHealth;
   infiniband_info?: {
      cluster_tag?: string;
      ports?: string[];
   };
   inv: number;
   ipmi_mac: MacAddress;
   ips?: string[];
   juggler_aggregate_name?: string;
   location: IHostLocation;
   macs: MacAddress[];
   messages?: IHostMessages;
   name: string;
   operation_state?: string;
   platform?: {
      system: string;
   };
   project: string;
   provisioner: Provisioner;
   restrictions?: Set<string>;
   scenario_id?: number;
   state: HostState;
   state_expire?: IHostStateExpire;
   state_reason?: string;
   status: string;
   status_audit_log_id?: string;
   status_author?: string;
   status_reason?: string;
   task?: {
      error?: string;
      status: string;
      status_message?: string;
   };
   ticket?: string;
   tier?: string;
   type?: string;
   uuid: string;
}

export type AddedHost = Pick<IHost, 'inv' | 'name' | 'state' | 'status' | 'status_author'>;

export interface IVlanConfig {
   expected_hbf_project_id: string;
   expected_native_vlan: number;
   expected_vlans: number[];
   hbf_project_id: string;
   hbf_project_id_synced: boolean;
   native_vlan: number;
   vlans: number[];
   vlans_synced: boolean;
}

export interface IVlanValue {
   items: Set<number>;
   selected?: number;
}

export interface IHostListRequest {
   config?: Set<string>;
   cursor?: number;
   fields: string[];
   health?: Set<string>;
   include_shadow?: boolean;
   limit: number;
   name?: string;
   physical_location?: Set<string>;
   project?: Set<string>;
   restrictions?: Set<string>;
   state?: Set<string>;
   status?: Set<string>;
   switch?: Set<string>;
   tags?: Set<string>;
   'sort-by'?: string;
   type?: Set<string>;
}

export interface IHostListPostRequest {
   invs: number[];
   names: string[];
   patterns: string[];
   uuids: string[];
}

export interface IHostActionOptions {
   check?: boolean;
   disable_admin_requests?: boolean;
   ignore_cms?: boolean;
   ignore_maintenance?: boolean;
   with_auto_healing?: boolean;
   erase_disks?: boolean;
}

export interface IHostActionParams {
   /**
    * Forcefully submit operation ignoring host maintenance status
    */
   ignore_maintenance: boolean;

   /**
    * An optional reason string
    */
   reason: string;
}

export interface IRemoveHostParams extends IHostActionParams {
   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * instant operation, don't schedule deletion task, don't ask CMS, etc.
    */
   instant: boolean; // false by default

   /**
    * delete the host also from LUI (for hosts provisioned by LUI)
    */
   lui: boolean; // false by default
}

export interface IConfigureDeployHostParams extends IHostActionParams {
   _deployConfig?: IDeployConfig;

   /**
    * Deploy config
    */
   config: string;

   /**
    * Modifying the configuration of the redeploying host action
    */
   deploy_config_policy: string;

   /**
    * Deploy host either in service or in project VLAN's
    */
   deploy_network: DeployNetwork;

   /**
    * Einstellung tags to apply to host before deployment
    * "pattern": "^[a-zA-Z0-9_:-]+$",
    * "maxItems": 20,
    */
   deploy_tags: Set<string>;

   /**
    * Provisioning system
    */
   provisioner: Provisioner;
}

export interface IEditExtraVlansHostParams extends IHostActionParams {
   _switchVlan: boolean;
   _vlanConfig?: IVlanValue;

   /**
    * VLAN IDs
    *
    * "maximum": 4094,
    * "minimum": 1,
    */
   vlans: Set<number>;
}

export interface IUpdateHostRestrictionParams extends IHostActionParams {
   restrictions: Set<string>;
}

export interface IMaintenanceProperties {
   _timeoutTime: Date | null;

   /**
    * CMS action name for switching to maintenance.
    *
    * Not used when adding hosts
    */
   cms_task_action?: CmsTaskAction;

   /**
    * Operation state of the host after switching to maintenance
    */
   operation_state: OperationState;

   /**
    * Power off the host
    */
   power_off: boolean;

   /**
    * Attach ticket to host's manual status.
    */
   ticket_key: string;

   /**
    * Status the host will be switched to at timeout_time
    */
   timeout_status: 'ready' | 'dead';

   /**
    * Time until which the specified status will be active for the host.
    * When the time comes, the host will be automatically switched to the status specified by timeout_status parameter.
    * It's guaranteed that status will be switched not earlier than the specified time,
    * but it's not guaranteed when exactly it will happen.
    * In practice, it will be in minutes since the specified time.
    */
   timeout_time: NullableDate | number;
}

export interface IAddHostsParams extends IHostActionParams {
   _deployConfig?: IDeployConfig;

   /**
    * Список хостов в поле
    */
   _hosts: Set<string>;

   _restrictionMode: 'default' | 'manual';

   /**
    * Check host after task completion and consider it as failed if host is not healthy
    */
   check: boolean; // true by default

   /**
    * Deploy config
    */
   config: string;
   /**
    * Modifying the configuration of the redeploying host action
    */
   deploy_config_policy: string;
   /**
    * Deploy host either in service or in project VLAN's
    */
   deploy_network: DeployNetwork;
   /**
    * Einstellung tags to apply to host before deployment
    *
    * "pattern": "^[a-zA-Z0-9_:-]+$",
    */
   deploy_tags: Set<string>;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean;

   /**
    * Check and setup DNS records for the host
    */
   dns: boolean;

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean;

   /**
    * instant operation, don't schedule deletion task, don't ask CMS, etc
    */
   instant: boolean;

   /**
    * Inventory number
    */
   inv?: number;

   maintenance_properties?: IMaintenanceProperties;

   /**
    * FQDN
    *
    * "pattern": "^([a-zA-Z0-9]+(-[a-zA-Z0-9]+)*\\.)+[a-z]{2,}$",
    */
   name?: string;

   /**
    * Project ID
    */
   project: string;

   /**
    * Provisioning system
    */
   provisioner: Provisioner;

   /**
    * Host restrictions
    */
   restrictions: Set<string>;

   /**
    * State that will be set for the host
    */
   state: 'free' | 'assigned' | 'maintenance';

   /**
    * Status that will be set for the host
    */
   status: 'ready' | 'manual' | 'dead';

   /**
    * Try to automatically repair the host if task will fail
    */
   with_auto_healing: boolean; // false by default (for API), true for UI
}

export interface HostPowerStatusResponse {
   is_powered_on: boolean;
}
