import { getValidResult, IValidationResult, validateEntity } from '@yandex-infracloud-ui/libs';
import { object } from 'yup';

import { Provisioner } from '../enums';
import { IConstants } from './dictionaries';
import { cmsShape, INewProject, isNewProjectValid, newProjectSchema } from './new_project';
import { CmsValue, DEFAULT_MAX_BUSY_HOSTS } from './projects';

const constants = {
   cms_api_versions: ['v1.3', 'v1.2', 'v1.1', 'v1.0'],
   provisioners: [Provisioner.EINE, Provisioner.LUI],
} as IConstants;

function getValidProject(): INewProject {
   return {
      automation_plot_id: undefined,
      bot_project_id: 456,
      cms: [{ url: 'default', max_busy_hosts: DEFAULT_MAX_BUSY_HOSTS }],
      deploy_config: 'test_deploy_config',
      id: 'test-project',
      name: 'Test project',
      provisioner: Provisioner.LUI,
      yc_iam_folder_id: undefined,
   };
}

function isFieldValid(field: keyof INewProject, value: any) {
   const result: IValidationResult<INewProject> = {
      errors: new Map(),
      isValid: true,
   };

   try {
      const context = { constants };

      newProjectSchema.validateSyncAt(field, { [field]: value }, { context });
   } catch (e) {
      result.errors.set(field, e.errors);
      result.isValid = false;
   }

   return result;
}

function createError(field: string, errors: string[]): IValidationResult<any> {
   return {
      errors: new Map([[field, errors]]),
      isValid: false,
   };
}

describe('new project validators', () => {
   it('should test project to be valid', () => {
      expect(isNewProjectValid(getValidProject(), constants)).toEqual(getValidResult());
   });

   describe('1 should validate project.bot_project_id', () => {
      it('should be valid', () => {
         expect(isFieldValid('bot_project_id', 454545454)).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('bot_project_id', null)).toEqual(
            createError('bot_project_id', ['bot_project_id is a required field']),
         );

         expect(isFieldValid('bot_project_id', -1)).toEqual(
            createError('bot_project_id', ['bot_project_id must be a positive integer']),
         );

         expect(isFieldValid('bot_project_id', 2.5)).toEqual(
            createError('bot_project_id', ['bot_project_id must be a positive integer']),
         );
      });
   });

   describe('1 should validate project.id', () => {
      it('should be valid', () => {
         expect(isFieldValid('id', 'xxx')).toEqual(getValidResult());
         expect(isFieldValid('id', 'xxx-yyy')).toEqual(getValidResult());
         expect(isFieldValid('id', 'x0123')).toEqual(getValidResult());
         expect(isFieldValid('id', 'x0123456789-x0123456789')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('id', 'x')).toEqual(createError('id', ['id must be at least 2 characters']));

         expect(isFieldValid('id', 'x0123456789-x0123456789-x0123456789-x0123456789')).toEqual(
            createError('id', ['id must be at most 32 characters']),
         );

         // noinspection LongLine
         const error = `id must have value like "qloud-common-dev-mtn", "yt-arnold-vla-masters". Regexp is /^[a-z][a-z\\d]*(?:-[a-z\\d]+)*$/`;

         expect(isFieldValid('id', 'xxx.yyy')).toEqual(createError('id', [error]));

         expect(isFieldValid('id', '0123')).toEqual(createError('id', [error]));

         expect(isFieldValid('id', 'XXXX')).toEqual(createError('id', [error]));
      });
   });

   describe('1 should validate project.name', () => {
      it('should be valid', () => {
         expect(isFieldValid('name', 'Any short name')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('name', '')).toEqual(createError('name', ['name is a required field']));

         expect(isFieldValid('name', 'x0123456789-x0123456789-x0123456789-x0123456789')).toEqual(
            createError('name', ['name must be at most 32 characters']),
         );
      });
   });

   describe('1 should validate project.deploy_tags', () => {
      it('should be valid', () => {
         expect(isFieldValid('deploy_tags', new Set())).toEqual(getValidResult());
         expect(isFieldValid('deploy_tags', new Set(['xxx', 'yyy']))).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         const items = [];
         for (let i = 0; i < 30; i += 1) {
            items.push(`tag-${i.toString()}`);
         }
         expect(isFieldValid('deploy_tags', new Set(items))).toEqual(
            createError('deploy_tags', ['deploy_tags field must have less than or equal to 20 items']),
         );

         expect(isFieldValid('deploy_tags', new Set(['?&&&:']))).toEqual(
            createError('deploy_tags', ['"?&&&:" is invalid tag. Tag must contain only [a-zA-Z0-9_=:-] symbols.']),
         );
      });
   });

   describe('should validate project.dns_domain', () => {
      it('should be valid', () => {
         expect(isFieldValid('dns_domain', 'valid-domain.ru')).toEqual(getValidResult());
         expect(isFieldValid('dns_domain', 'sub.valid-domain.ru')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('dns_domain', 'invalid-domain.x')).toEqual(
            createError('dns_domain', ['dns_domain must look like "valid-domain.ru"']),
         );
      });
   });

   describe('should validate project.hbf_project_id', () => {
      it('should be valid', () => {
         expect(isFieldValid('hbf_project_id', '0x12FC')).toEqual(getValidResult());
         expect(isFieldValid('hbf_project_id', '0x12fc')).toEqual(getValidResult());
         expect(isFieldValid('hbf_project_id', '12FC')).toEqual(getValidResult());
         expect(isFieldValid('hbf_project_id', '12fc')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('hbf_project_id', 'xyz1')).toEqual(
            createError('hbf_project_id', ['hbf_project_id must contain HEX number like "12FC" or "0x12FC"']),
         );
      });
   });

   describe('should validate project.owned_vlans', () => {
      it('should be valid', () => {
         expect(isFieldValid('owned_vlans', [])).toEqual(getValidResult());
         expect(isFieldValid('owned_vlans', [344, 555])).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('owned_vlans', [12, -3])).toEqual(
            createError('owned_vlans', ['VLAN must be between 1 and 4094']),
         );
      });
   });

   describe('1 should validate project.owners', () => {
      it('should be valid', () => {
         expect(isFieldValid('owners', new Set())).toEqual(getValidResult());
         expect(isFieldValid('owners', new Set(['khoden']))).toEqual(getValidResult());
         expect(isFieldValid('owners', new Set(['@khoden']))).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         const error = createError('owners', ['owners must contain only nicknames, e.g. "login" or "@service"']);

         expect(isFieldValid('owners', new Set(['khoden@yandex-team.ru']))).toEqual(error);

         // noinspection LongLine
         expect(
            isFieldValid(
               'owners',
               new Set(['khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789']),
            ),
         ).toEqual(error);

         // noinspection LongLine
         expect(
            isFieldValid(
               'owners',
               new Set([
                  '@khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789',
               ]),
            ),
         ).toEqual(error);
      });
   });

   describe('1 should validate project.provisioner', () => {
      it('should be valid', () => {
         expect(isFieldValid('provisioner', Provisioner.LUI)).toEqual(getValidResult());
         expect(isFieldValid('provisioner', Provisioner.EINE)).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('provisioner', '')).toEqual(
            createError('provisioner', ['provisioner is a required field']),
         );

         expect(isFieldValid('provisioner', 'xxx')).toEqual(
            createError('provisioner', ['provisioner must be one of supported provisioners']),
         );
      });
   });

   describe('1 should validate project.tags', () => {
      it('should be valid', () => {
         expect(isFieldValid('tags', new Set())).toEqual(getValidResult());
         expect(isFieldValid('tags', new Set(['xxx', 'yyy']))).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('tags', new Set(['?&&&:']))).toEqual(
            createError('tags', ['"?&&&:" is invalid tag. Tag must contain only [a-zA-Z0-9_.:-] symbols.']),
         );
      });
   });
});

describe('1 cms value validators', () => {
   const schema = object().shape(cmsShape);

   function isCmsValid(v: CmsValue): IValidationResult<CmsValue> {
      return validateEntity(schema, Object.keys(cmsShape) as (keyof CmsValue)[], v, { constants });
   }

   describe('default CMS', () => {
      it('should be valid', () => {
         expect(isCmsValid({ url: 'default', max_busy_hosts: 1 })).toEqual(getValidResult());
      });

      it('should be valid with max_busy_hosts greater then 1', () => {
         expect(isCmsValid({ url: 'default', max_busy_hosts: 422 })).toEqual(getValidResult());
      });

      it('should be invalid without max_busy_hosts', () => {
         expect(isCmsValid({ url: 'default', api_version: 'v1.3' })).toEqual(
            createError('max_busy_hosts', ['max_busy_hosts is a required field']),
         );
      });

      it('should be invalid if max_busy_hosts less then 1', () => {
         expect(isCmsValid({ url: 'default', max_busy_hosts: 0 })).toEqual(
            createError('max_busy_hosts', ['max_busy_hosts must be greater than or equal to 1']),
         );
      });
   });

   describe('custom CMS', () => {
      const validCmsUrl = 'http://clusterstate.yandex-team.ru/api/v1';

      it('should be valid with valid version', () => {
         expect(isCmsValid({ api_version: 'v1.0', tvm_app_id: 222, url: validCmsUrl })).toEqual(getValidResult());
      });

      it('should be invalid without tvm_app_id in not YP version', () => {
         expect(isCmsValid({ url: validCmsUrl, api_version: 'v1.3' })).toEqual(
            createError('tvm_app_id', ['tvm_app_id is a required field']),
         );
      });

      it('should be invalid with invalid url', () => {
         expect(isCmsValid({ url: 'xxsdfyyy', tvm_app_id: 222, api_version: 'v1.3' })).toEqual(
            createError('url', ['url must be valid url or "default"']),
         );
      });

      it('should be invalid with wrong api_version', () => {
         expect(isCmsValid({ url: validCmsUrl, tvm_app_id: 222, api_version: 'x' })).toEqual(
            createError('api_version', ['api_version has unsupported CMS version']),
         );
      });
   });
});
