import { CMS_URL_REGEXP, DNS_DOMAIN_REGEXP, MTN_PROJECT_ID_REGEXP, OWNER_REGEXP, PROJECT_ID_REGEXP } from '../regexps';
import { Provisioner } from '../enums';
import { deployTagSchema, projectTagSchema } from '../validation';
import { number, object, Schema, string, array } from 'yup';
import { setAsArray, validateEntity } from '@yandex-infracloud-ui/libs';

import { IConstants } from './dictionaries';
import { IProject } from './projects';

export interface IDeployConfig {
   config: string;
   provisioner: Provisioner;
}

/**
 * @deprecated
 */
export interface INewProject extends Partial<IProject> {
   enable_dns_automation?: boolean;
   enable_healing_automation?: boolean;
   reason?: string;
}

/**
 * @deprecated
 */
export function isHbfProjectIdValid(id: string): boolean {
   return MTN_PROJECT_ID_REGEXP.test(id);
}

function checkProvisioner(v: Provisioner) {
   // @ts-ignore
   const constants: IConstants = this.options.context.constants;

   return constants.provisioners.includes(v);
}

function checkCmsVersion(v: string) {
   // @ts-ignore
   const constants: IConstants = this.options.context.constants;

   return constants.cms_api_versions.includes(v);
}

/* eslint-disable no-template-curly-in-string */
export const cmsShape = {
   api_version: string().when('url', {
      is: 'default',
      otherwise: string().required().test('cmsVersion', '${path} has unsupported CMS version', checkCmsVersion),
      then: string().notRequired(),
   }),
   max_busy_hosts: number().when('url', {
      is: 'default',
      otherwise: number().notRequired(),
      then: number().required().min(1),
   }),
   tvm_app_id: number().when(['url', 'api_version'], {
      is: (url: string) => url !== 'default',
      otherwise: number().notRequired().positive(),
      then: number().required().positive(),
   }),
   url: string().required().matches(CMS_URL_REGEXP, '${path} must be valid url or "default"'),
};

/* eslint-disable no-template-curly-in-string */
// noinspection LongLine
const newProjectShape = {
   bot_project_id: number()
      .required()
      .typeError('${path} is a required field')
      .integer('${path} must be a positive integer')
      .positive('${path} must be a positive integer'),

   cms: array().of(object().shape(cmsShape)),

   deploy_config: string().required(),

   deploy_tags: deployTagSchema,

   dns_domain: string().matches(DNS_DOMAIN_REGEXP, '${path} must look like "valid-domain.ru"'),

   hbf_project_id: string().matches(MTN_PROJECT_ID_REGEXP, '${path} must contain HEX number like "12FC" or "0x12FC"'),

   id: string()
      .required()
      .min(2)
      .max(32)
      .matches(
         PROJECT_ID_REGEXP,
         `$\{path} must have value like "qloud-common-dev-mtn", "yt-arnold-vla-masters". ` +
            `Regexp is ${PROJECT_ID_REGEXP}`,
      ),

   name: string().required().min(1).max(32),

   owned_vlans: setAsArray().of(number().min(1, 'VLAN must be between 1 and 4094').max(4094)) as any,

   owners: setAsArray().of(
      string().matches(OWNER_REGEXP, 'owners must contain only nicknames, e.g. "login" or "@service"'),
   ) as any, // FIXME typings

   provisioner: string()
      .required()
      .test('provisioner', '${path} must be one of supported provisioners', checkProvisioner) as Schema<Provisioner>,

   tags: projectTagSchema,

   yc_iam_folder_id: string(),
};

export const newProjectSchema = object().shape<Partial<INewProject>>(newProjectShape);

export const isNewProjectValid = (project: Partial<INewProject>, constants: IConstants) =>
   validateEntity(newProjectSchema, Object.keys(newProjectShape) as (keyof INewProject)[], project, { constants });
