import { getValidResult, IValidationResult } from '@yandex-infracloud-ui/libs-next';

import { Provisioner } from '../enums';

import { addPreorderParamsSchema, IAddPreorderParams, isAddPreorderParamsValid } from './preorders';

function getValidParams(): Partial<IAddPreorderParams> {
   return {
      id: 1,
      project: 'test',
   };
}

function isFieldValid(field: keyof IAddPreorderParams, value: any) {
   const result: IValidationResult<IAddPreorderParams> = {
      errors: new Map(),
      isValid: true,
   };

   try {
      const context = undefined;

      addPreorderParamsSchema.validateSyncAt(field, { [field]: value }, { context });
   } catch (e) {
      result.errors.set(field, e.errors);
      result.isValid = false;
   }

   return result;
}

function createError(field: string, errors: string[]): IValidationResult<any> {
   return {
      errors: new Map([[field, errors]]),
      isValid: false,
   };
}

describe('add preorder validators', () => {
   it('should test params to be valid', () => {
      expect(isAddPreorderParamsValid(getValidParams())).toEqual(getValidResult());
   });

   describe('should validate id', () => {
      it('should be valid', () => {
         expect(isFieldValid('id', 20)).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('id', 0)).toEqual(createError('id', ['id must be greater than or equal to 1']));

         expect(isFieldValid('id', -1)).toEqual(createError('id', ['id must be greater than or equal to 1']));

         expect(isFieldValid('id', null)).toEqual(createError('id', ['id must be a number']));
      });
   });

   describe('should validate project', () => {
      it('should be valid', () => {
         expect(isFieldValid('project', 'test')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('project', '')).toEqual(createError('project', ['project is a required field']));

         expect(isFieldValid('project', null)).toEqual(createError('project', ['project should be project_id']));
      });
   });

   describe('should validate deploy_config', () => {
      it('should be valid', () => {
         expect(isFieldValid('deploy_config', 'ubuntu-16.04')).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('deploy_config', null)).toEqual(
            createError('deploy_config', ['deploy_config must be a config name']),
         );
      });
   });

   describe('should validate provisioner', () => {
      it('should be valid', () => {
         expect(isFieldValid('provisioner', Provisioner.EINE)).toEqual(getValidResult());

         expect(isFieldValid('provisioner', Provisioner.LUI)).toEqual(getValidResult());
      });

      it('should be invalid', () => {
         expect(isFieldValid('provisioner', 'xxxx')).toEqual(
            createError('provisioner', ['provisioner must be one of the following values: lui, eine']),
         );
      });
   });
});
