import { setAsArray, validateEntity } from '@yandex-infracloud-ui/libs-next';
import { bool, number, object, string } from 'yup';
import { Provisioner } from '../enums';
import { IDeployConfig } from './new_project';

export const allPreorderFields = [
   'acquired_hosts',
   'deploy_config',
   'errors',
   'failed_hosts',
   'id',
   'issuer',
   'messages',
   'owner',
   'prepare',
   'processed',
   'project',
   'provisioner',
   'restrictions',
];

export interface IPreorder {
   acquired_hosts: number[];
   errors?: string[];
   failed_hosts: number[];
   id: number;
   issuer: string;
   messages?: string[];
   owner: string;
   prepare: boolean;
   processed: boolean;
   project: string;
}

export interface IPreorderBaseParams {
   /**
    * An optional reason string
    */
   reason?: string;

   /**
    * Use admin privileges to process the request
    */
   sudo?: boolean;
}

export interface IAddPreorderParams extends IPreorderBaseParams {
   _deployConfig?: IDeployConfig;
   _restrictionMode: 'default' | 'manual';

   /**
    * Config to deploy the hosts with
    */
   deploy_config?: string;

   /**
    * Preorder ID
    */
   id: number;

   /**
    * Prepare taken hosts
    */
   prepare?: boolean;

   /**
    * Project ID to add the hosts to
    */
   project: string;

   /**
    * Provisioning system
    */
   provisioner?: Provisioner;

   /**
    * Restrictions to set for the hosts
    */
   restrictions?: Set<string>;
}

/* eslint-disable no-template-curly-in-string */
const addPreorderParamsShape = {
   deploy_config: string().typeError('${path} must be a config name'),
   id: number().typeError('${path} must be a number').required().min(1),
   prepare: bool(),
   project: string().typeError('${path} should be project_id').required(),
   provisioner: string().oneOf([Provisioner.LUI, Provisioner.EINE]) as any,
   reason: string(),
   restrictions: setAsArray().of(string()) as any,
   sudo: bool(),
};

export const addPreorderParamsSchema = object().shape<Partial<IAddPreorderParams>>(addPreorderParamsShape);

export const isAddPreorderParamsValid = (params: Partial<IAddPreorderParams>) =>
   validateEntity(
      addPreorderParamsSchema,
      Object.keys(addPreorderParamsShape) as (keyof IAddPreorderParams)[],
      params,
      null,
   );
