import { AutomationLimit } from '../AutomationLimit';
import { DNS_DOMAIN_REGEXP, PROJECT_ID_REGEXP } from '../regexps';
import { DeployNetwork, ProjectType, Provisioner } from '../enums';
import { IDeployConfig } from './new_project';
import { IProjectReports } from '../../rich_shared/project_report_settings';
import { TimeoutMode, TimeoutValue, timeoutValueSchema } from './common';
import { isEmpty, mergeSets } from '@yandex-infracloud-ui/libs';
import { number, object, string } from 'yup';
import { DnsZoneFormParams } from 'state/fullProjectForm';

export const allProjectFields = [
   'all_available_project_checks',
   'automation_limits',
   'automation_plot_id',
   'bot_project_id',
   'cauth_settings.flow_type',
   'cauth_settings.trusted_sources',
   'cauth_settings.key_sources',
   'cauth_settings.secure_ca_list_url',
   'cauth_settings.insecure_ca_list_url',
   'cauth_settings.sudo_ca_list_url',
   'cauth_settings.krl_url',
   'certificate_deploy',
   'cms_settings',
   'default_host_restrictions',
   'deploy_config',
   'deploy_config_policy',
   'deploy_network',
   'deploy_tags',
   'dns_automation.credit',
   'dns_automation.credit_end_time',
   'dns_automation.enabled',
   'dns_automation.status_message',
   'dns_domain',
   'extra_vlans',
   'fsm_handbrake.audit_log_id',
   'fsm_handbrake.issuer',
   'fsm_handbrake.reason',
   'fsm_handbrake.ticket_key',
   'fsm_handbrake.timeout_time',
   'fsm_handbrake.timestamp',
   'hbf_project_id',
   'healing_automation.credit',
   'healing_automation.credit_end_time',
   'healing_automation.enabled',
   'healing_automation.status_message',
   'host_limits',
   'id',
   'maintenance_plot_id',
   'manually_disabled_checks',
   'name',
   'native_vlan',
   'notifications.recipients',
   'owned_vlans',
   'owners',
   'profile',
   'profile_tags',
   'provisioner',
   'reboot_via_ssh',
   'reports',
   'roles',
   'tags',
   'tier',
   'type',
   'vlan_scheme',
   'yc_iam_folder_id',
   'yc_dns_zone',
   'yc_dns_zone_id',
];

export interface IAutomationLimits {
   [key: string]: AutomationLimit[];
}

interface AutomationCredit {
   [key: string]: number;
}

export interface Automation {
   credit?: AutomationCredit;
   credit_end_time?: number;
   enabled: boolean;
   status_message?: string;
}

export function getTumblerValue(automation?: Automation): boolean {
   return automation ? automation.enabled : false;
}

export enum NotificationSeverity {
   Bot = 'bot',
   Audit = 'audit',
   Info = 'info',
   Warning = 'warning',
   Error = 'error',
   Critical = 'critical',
}

/**
 * Recipients by severity
 */
export type NotificationRecipients = {
   [severity in NotificationSeverity]: string[];
};

export function createEmptyNotificationRecipients(): NotificationRecipients {
   return {
      [NotificationSeverity.Audit]: [],
      [NotificationSeverity.Bot]: [],
      [NotificationSeverity.Critical]: [],
      [NotificationSeverity.Error]: [],
      [NotificationSeverity.Info]: [],
      [NotificationSeverity.Warning]: [],
   };
}

export const DEFAULT_MAX_BUSY_HOSTS = 10;

export const DEFAULT_CERTIFICATE_DEPLOY = false;

export interface IDefaultCms {
   /**
    * Maximum number of hosts that default CMS will allow to process concurrently
    * "minimum": 1,
    */
   max_busy_hosts: number;

   /**
    * CMS type
    */
   url: 'default';
}

export interface ICustomCms {
   /**
    * Supported version of CMS API protocol
    */
   api_version: string;

   /**
    * TVM application id of CMS
    *
    * Optional for admins, required for users. Disabled if api_version = YP.
    */
   tvm_app_id?: number;

   /**
    * Custom CMS URL
    * "^https?://(?:[a-z0-9]+(?:-[a-z0-9]+)*\\.)+[a-z]{2,}(?::[0-9]+)?(?:/(?:[-._0-9a-zA-Z]+/)*(?:[-._0-9a-zA-Z]+)?)*?$"
    */
   url: string;

   temporary_unreachable_enabled?: boolean;
}

export type CmsValue = IDefaultCms | ICustomCms;

export enum AutomationType {
   DNS = 'dns',
   Healing = 'healing',
}

export enum CAuthFlowType {
   Classic = 'classic',
   BackendSource = 'backend_sources',
}

export enum CAuthKeySources {
   Staff = 'staff',
   Secure = 'secure',
   Insecure = 'insecure',
}

export enum CAuthTrustedSources {
   Walle = 'walle',
   Yp = 'yp',
   Idm = 'idm',
   Ycloud = 'ycloud',
   Hd = 'hd',
   Bot = 'bot',
   Conductor = 'conductor',
   IdmConductor = 'idm-conductor',
   Cms = 'cms',
   IdmCms = 'idm-cms',
}

export interface ICAuthSettings {
   flow_type?: CAuthFlowType;
   insecure_ca_list_url?: string;
   key_sources?: CAuthKeySources[];
   krl_url?: string;
   secure_ca_list_url?: string;
   sudo_ca_list_url?: string;
   trusted_sources?: CAuthTrustedSources[];
}

export interface IAutomationCredits {
   time: number;

   [check: string]: number;
}

export interface IAutomationForm {
   reason: string;
}

export interface IAutomationCreditsForm extends IAutomationForm {
   credit: IAutomationCredits;
}

export interface IDnsAutomationCreditsForm extends IAutomationCreditsForm {
   dns_domain: string;
}

export interface IDnsDomainForm extends IAutomationForm {
   dns_domain: string;
   yc_dns_zone_id: string;
}

export interface DnsAutomationForm extends IAutomationForm {
   credit_end_time: string;
   credit: AutomationCredit;
}

export type HealingAutomationForm = DnsAutomationForm;

export interface IFsmHandbrakeForm extends IAutomationForm {
   ticket_key: string | null;
   timeout: TimeoutValue;
}

const reasonSchema = string().label('Comment');

const creditSchema = object<IAutomationCredits>({
   time: number()
      .label('Credit time')
      // eslint-disable-next-line no-template-curly-in-string
      .moreThan(-1, '${path} must be a positive or zero number of seconds'),
});

const dnsDomainSchema = string().label('DNS-automation domain').matches(DNS_DOMAIN_REGEXP);
const dnsZoneIdSchema = string().label('DNS Zone ID');

export const automationFormValidationSchema = object<IAutomationForm>({
   reason: reasonSchema,
});

export const automationCreditsFormValidationSchema = object<IAutomationCreditsForm>({
   credit: creditSchema,
   reason: reasonSchema,
});

export const dnsAutomationCreditsFormValidationSchema = object<IDnsAutomationCreditsForm>({
   credit: creditSchema,
   dns_domain: dnsDomainSchema.required(),
   reason: reasonSchema,
});

export const dnsFormValidationSchema = object<IDnsDomainForm>({
   dns_domain: dnsDomainSchema,
   yc_dns_zone_id: dnsZoneIdSchema,
   reason: reasonSchema,
});

export const fsmHandbrakeFormValidationSchema = object<IFsmHandbrakeForm>({
   reason: reasonSchema,
   ticket_key: string().label('Ticket').required().nullable(),
   timeout: timeoutValueSchema,
});

export interface IProjectSubscribers {
   recipients: NotificationRecipients;
}

export enum VlanSchema {
   // noinspection JSUnusedGlobalSymbols
   Cloud = 'cloud',

   /**
    * В этой схеме ip_method='mac'
    */
   MTN = 'mtn',

   /**
    * Особая схема mtn, в которой ip_method='hostname'
    *
    * С точки зрения UI виртуальная, т.к. при загрузке проекта с такой схемой она заменяется на mtn
    */
   MTNHostId = 'mtn-hostid',
   None = 'none',
   Search = 'search',
   Static = 'static',
}

export interface IdmRequest {
   id: number;
   login: string;
}

export interface FsmHandbrakePayload {
   audit_log_id?: string;
   issuer?: string;
   reason?: string;
   ticket_key?: string;

   /**
    * Дедлайн (метка в секундах)
    */
   timeout_time: number;

   /**
    * Дата включения FSM-handbrake (метка в секундах)
    */
   timestamp?: number;
}

export interface ProjectOwners {
   /**
    * Список действующих (подтвержденных владельцев) минус те, кто был удален крестиком (toRemove)
    */
   actualOwners: Set<string>;

   /**
    * Список запрошенных, но неподтвержденных еще в IDM овнеров. Приходит с сервера.
    */
   requestedOwners: Map<string, IdmRequest>;

   /**
    * Список отозванных в IDM овнеров (но еще не удаленных окончательно)
    */
   revokingOwners: Map<string, IdmRequest>;

   /**
    * Список овнеров, которые еще предстоит запросить в IDM
    */
   toAdd: Set<string>;

   /**
    * Список овнеров, доступ которых нужно отозвать
    */
   toRemove: Set<string>;
}

export function createProjectOwners(seed?: Partial<ProjectOwners>): ProjectOwners | undefined {
   const revokingOwners = seed && seed.revokingOwners ? seed.revokingOwners : new Map<string, IdmRequest>();

   const actualOwners = seed && seed.actualOwners ? seed.actualOwners : new Set();

   // Убираю уже отозванных пользователей из списка актуальных
   revokingOwners.forEach(o => actualOwners.delete(o));

   return {
      actualOwners,

      requestedOwners: seed && seed.requestedOwners ? seed.requestedOwners : new Map<string, IdmRequest>(),

      revokingOwners,

      toAdd: seed && seed.toAdd ? seed.toAdd : new Set(),

      toRemove: seed && seed.toRemove ? seed.toRemove : new Set(),
   } as ProjectOwners;
}

export function isProjectOwnersEmpty(v: ProjectOwners): boolean {
   return (
      isEmpty(v.actualOwners) &&
      isEmpty(v.requestedOwners) &&
      isEmpty(v.revokingOwners) &&
      isEmpty(v.toAdd) &&
      isEmpty(v.toRemove)
   );
}

export interface ProjectRoles {
   [role: string]: string[];
}

export interface ProjectCmsSettings {
   cms?: string; // string for getting project, CmsValue for editing sic!
   cms_api_version?: string;
   cms_max_busy_hosts?: number;
   cms_tvm_app_id?: number;
   temporary_unreachable_enabled?: boolean;
}

export interface IProjectBoxes {
   dns?: string;
   eine?: string;
}

export interface IProject {
   _checks?: IAutomationLimits;
   _deployConfig?: IDeployConfig;
   _owners?: ProjectOwners;
   all_available_project_checks: string[];
   automation_limits?: IAutomationLimits;
   automation_plot_id?: string;
   bot_project_id?: number;
   boxes?: IProjectBoxes;
   cauth_settings?: ICAuthSettings;
   cauth_flow_type?: CAuthFlowType;
   cauth_insecure_ca_list_url?: string;
   cauth_key_sources?: CAuthKeySources[];
   cauth_krl_url?: string;
   cauth_secure_ca_list_url?: string;
   cauth_sudo_ca_list_url?: string;
   cauth_trusted_sources?: CAuthTrustedSources[];
   certificate_deploy?: boolean;
   cms?: CmsValue[]; // string for getting project, CmsValue for editing sic!
   cms_settings?: ProjectCmsSettings[] | CmsValue[];
   default_host_restrictions?: Set<string>;
   deploy_config?: string;
   deploy_config_policy?: string;
   deploy_network?: DeployNetwork;
   deploy_tags?: Set<string>;
   dns_automation?: Automation;
   dns_domain?: string;
   dnsZone?: DnsZoneFormParams;
   extra_vlans?: Set<number>;
   fsm_handbrake?: FsmHandbrakePayload;
   hbf_project_id?: string;
   healing_automation?: Automation;
   host_limits?: IAutomationLimits;
   id: string;
   ip_method?: 'mac' | 'hostname';
   maintenance_plot_id?: string;
   name: string;
   native_vlan?: number;
   notifications?: IProjectSubscribers;
   manually_disabled_checks: Set<string>;
   owned_vlans?: Set<number>;
   owners?: Set<string>;
   profile?: string;
   profile_tags?: Set<string>;
   provisioner?: Provisioner;
   reboot_via_ssh?: boolean;
   reports?: IProjectReports;
   roles?: ProjectRoles;
   tags?: Set<string>;
   tier?: number;
   type?: ProjectType;
   vlan_scheme?: VlanSchema;
   use_fastbone: boolean;
   yc_iam_folder_id?: string;
   yc_dns_zone?: { domain: string; link: string };
   yc_dns_zone_id?: string;
}

export interface IProjectListRequest {
   fields: string[];
}

export interface IProjectActionParams {
   /**
    * An optional reason string
    */
   reason?: string;
}

export interface IdmRequest {
   id: number;
   login: string;
}

export interface ICloneProjectParams extends IProjectActionParams {
   /**
    * A unique ID
    *
    * "maxLength": 32,
    * "pattern": "^[a-z][a-z0-9]*(?:-[a-z0-9]+)*$",
    */
   id: string;

   /**
    * Name that will identify the project in UI
    *
    * "maxLength": 32, "minLength": 1,
    */
   name: string;
}

/**
 * @deprecated use IFsmHandbrakeForm
 */
export interface IFsmHandbrakeProjectParams extends IProjectActionParams {
   _mode: TimeoutMode;

   /**
    * Ticket for project's emergency.
    */
   ticket_key: string;

   /**
    * Timeout for which the emergency brake will be active for the project.
    * This is an alternative way of setting timeout time.
    * The timeout_time parameter has a priority.
    *
    * "maximum": 604800, "minimum": 600,
    */
   timeout?: number;

   /**
    * Time until which the emergency brake will be active for the project.
    * When the time comes, the emergency brake will be automatically turned off by wall-e.
    * It's guaranteed that status will be switched not earlier than the specified time.
    * This is an alternative way of setting timeout which has a priority over the timeout parameter.
    */
   timeout_time?: Date | number;
}

export function isProjectCloneParamsValid(params: ICloneProjectParams): boolean {
   const id = params.id.trim();
   if (!id || id.length > 32 || !PROJECT_ID_REGEXP.test(id)) {
      return false;
   }

   const name = params.name.trim();
   // noinspection RedundantIfStatementJS
   if (!name || name.length > 32) {
      return false;
   }

   return true;
}

const auditLogTypesForDns = new Set([
   'disable-dns-automation',
   'enable-dns-automation',
   'disable-automation',
   'enable-automation',
]);

const auditLogTypesForHealing = new Set([
   'disable-automated-healing',
   'enable-automated-healing',
   'disable-automation',
   'enable-automation',
]);

const auditLogTypesForFsm = new Set(['extend-fsm-handbrake', 'fsm-handbrake']);

export const auditLogTypesForAutomations = mergeSets(auditLogTypesForDns, auditLogTypesForHealing, auditLogTypesForFsm);
