import { SystemUser, toQuery } from '@yandex-infracloud-ui/libs';
import { BehaviorSubject } from 'rxjs';

import idmPng from '../design/avatars/idm.png';
import wallePng from '../design/avatars/wall-e.png';
import { UserModel } from './api_models/auth';
import { Provisioner } from './enums';

export const FAST_UPDATE_INTERVAL = 5000;

export const SLOW_UPDATE_INTERVAL = 30000;

export const NONE_ITEM = 'none';

export const SYSTEM_USERS = new Map<string, SystemUser>([
   ['wall-e', { login: 'wall-e', avatar: wallePng }],
   ['idm', { login: 'idm', avatar: idmPng }],
]);

export const HIDDEN_PROJECT_ROLES = new Set(['ssh_rebooter']);

export const CURSOR_PAGINATION_LIMIT = 100;

function _getParams(params: any) {
   return params ? '?' + toQuery(params) : '';
}

/**
 * Вынес все роуты в одно место, чтобы не хранить в одним разделе приложения знания о суброутах другого раздела.
 *
 * Простая ссылка на другой раздел тоже работает,
 * но приводит к лишним редиректам с индексного адреса на адрес по-умолчанию.
 */
export const ROUTE_LINKS = {
   auditLog(params?: any) {
      return `/projects/log${_getParams(params)}`;
   },
   auditLogItem(id: string) {
      return `/projects/log/${id}`;
   },
   operationLog(params?: any) {
      return `/projects/operation_log${_getParams(params)}`;
   },
   operationLogItem(id: string) {
      return `/projects/operation_log/${id}`;
   },
   automationPlot(id: string) {
      return `/automation_plots/${id}`;
   },
   automationPlots() {
      return '/automation_plots';
   },
   maintenancePlot(id: string) {
      return `/maintenance_plots/${id}`;
   },
   maintenancePlots() {
      return '/maintenance_plots';
   },
   hostList(params?: any) {
      return `/projects/hosts${_getParams(params)}`;
   },
   home(params?: any) {
      return this.hostList(params);
   },
   project(project: string) {
      return `/project/${project}`;
   },
   projectNew() {
      return '/project/new';
   },
   preorders() {
      return '/preorders';
   },
   preorder(id: number) {
      return `/preorders/${id}`;
   },
   host(host: string, inv?: number) {
      return `/host/${host || inv}`;
   },
   hostAuditLog(host: string, id?: string) {
      return `/host/${host}/audit_log/${_getParams({ id })}`;
   },
   hostOperationLog(host: string, params?: any) {
      return `/host/${host}/operation_log${_getParams(params)}`;
   },
   hostRestrictions(host: string) {
      return `/host/${host}/restrictions`;
   },
   hostHealth(host: string) {
      return `/host/${host}/health`;
   },
   scenarios(workStatus?: Set<string>) {
      return `/scenarios${_getParams({ work_status: workStatus })}`;
   },
   scenario(id: number): string {
      return `/scenarios/${id}`;
   },
};

export const EXTERNAL_LINKS = {
   deployConfig(provisioner: Provisioner, config: string) {
      return provisioner === Provisioner.EINE
         ? `//eine.yandex-team.ru/profiles/${config}`
         : `//setup.yandex-team.ru/#!/configs/view/${config}`;
   },
   passport(retpath: string) {
      const params = { mode: 'auth', retpath };

      return `//passport.yandex-team.ru/passport${_getParams(params)}`;
   },
   bot(inv: number) {
      return ` https://bot.yandex-team.ru/umka/assets/${inv}`;
   },
   botPreorder(preorderId: string) {
      return `https://bot.yandex-team.ru/hwr/preorders/${preorderId}`;
   },
   coronerLog(host: string) {
      return `http://test-coroner-7.sas.yp-c.yandex.net:8080/api/v1/raw${_getParams({ host })}`;
   },
   eine(hostName: string, host: string = 'ro.eine.yandex-team.ru') {
      return `https://${host}/computers/hostname/${hostName}`;
   },
   lui(hostName: string, host: string = 'setup.yandex-team.ru') {
      return `https://${host}/#!/servers/view/${hostName}`;
   },
   juggler(hostName: string) {
      return `https://juggler.yandex-team.ru/raw_events/?query=host%3D${hostName}&limit=40`;
   },
   jugglerAggregationChecks(jugglerAggregateName: string) {
      return `https://juggler.yandex-team.ru/aggregate_checks/?query=host%3D${jugglerAggregateName}`;
   },
   inv(inv: number, host: string = 'setup.yandex-team.ru') {
      return `https://${host}/ipmiproxy/api/v1.0/${inv}/kvm`;
   },
   walleDocs() {
      return 'https://docs.yandex-team.ru/wall-e/';
   },
   walleDocsGuide(guide: string) {
      return `https://docs.yandex-team.ru/wall-e/guide/${guide}`;
   },
   walleDraft(name: string) {
      return `https://wiki.yandex-team.ru/wall-e/devel/drafts/${name}`;
   },
   rackTables(switch1: string, port: string) {
      return `https://racktables.yandex.net/index.php?page=search&t=8021qports&port=${port}&q=${switch1}`;
   },
   abc(slug: string) {
      return `https://abc.yandex-team.ru/services/${slug}/`;
   },
   uiChangelog() {
      return 'https://wiki.yandex-team.ru/wall-e/ui-changelog/';
   },
   sentry() {
      return 'https://sentry.t.yandex-team.ru/monitoring/wall-e-ui/';
   },
   yasm(project: string) {
      return `https://yasm.yandex-team.ru/template/panel/wall-e-metrics/project=${project}/`;
   },
   rtc(hostName: string) {
      return `https://rtc-lookup.in.yandex-team.ru/host/${hostName}`;
   },
};

/**
 * "Глобальные" переменные приложения
 *
 * Был вынужден вынести из сервиса auth во избежание циклической зависимости:
 * auth -> dictApi -> baseApi -> auth
 */
export const globalState = {
   csrfToken: new BehaviorSubject<string>(''),
   user: new BehaviorSubject<UserModel | null>(null),
};
