import { BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs-next';
import { forkJoin, Observable } from 'rxjs';
import { map, switchMap, tap } from 'rxjs/operators';

import { dictApi, sentry } from '../../services';
import { IABCService, IABCServiceResponse } from './models';

class AbcBotApi extends BaseApi {
   /**
    * Сложный код, т.к. нужно прописать для каждого сервиса bot_project_id
    */
   public findABCService(query: string): Observable<IABCService[]> {
      const params = {
         fields: ['id', 'name.en', 'state', 'slug'],
         page_size: 10,
         slug__contains: query,
         state__in: ['develop', 'supported', 'needinfo'],
      };

      const itemsObs = this.request<typeof params, void, IABCServiceResponse>(HttpMethod.GET, 'services/', params).pipe(
         map(resp => resp.results),
      );

      const botProjectMapObs = dictApi.getBotProjectMap();

      return forkJoin(itemsObs, botProjectMapObs).pipe(
         // Fill bot_project_id
         map(([items, botProjectMap]) => {
            items.forEach(service => {
               if (!botProjectMap.has(service.id)) {
                  console.warn(`Can not find bot project for ABC service #${service.id} "${service.name.en}"`);
               }

               // -1 - BOT проект не найден
               // Не можем использовать undefined/null из-за конфликтов с значениями при незаполненной формой
               service.bot_project_id = botProjectMap.get(service.id) ? botProjectMap.get(service.id)! : -1;
            });

            return items;
         }),
      );
   }

   public getABCServiceByBotProject(botProjectId: number): Observable<IABCService> {
      return dictApi.getAbcServiceProjectMap().pipe(
         map(abcServiceMap => abcServiceMap.get(botProjectId)),
         // TODO отработать отсутствие соответствия
         switchMap(abcServiceId => this._getABCServiceById(abcServiceId!, botProjectId)),
      );
   }

   protected getCSRF(): string {
      return ''; // нерелевантно для данного API
   }

   protected handleError(resp: Response, error: IApiError): void {
      sentry.sendApiError(resp, error);
   }

   private _getABCServiceById(id: number, botProjectId: number): Observable<IABCService> {
      const params = {
         fields: ['id', 'name.en', 'state', 'slug'],
         id,
         page_size: 10,
      };

      return this.request<typeof params, void, IABCServiceResponse>(
         HttpMethod.GET,
         '/external_api/abc/services/',
         params,
      ).pipe(
         map(resp => resp.results[0]),
         tap(service => (service.bot_project_id = botProjectId)),
      );
   }
}

export const abcBotApi = new AbcBotApi('/external_api/abc');
