import { isEqual, PER_PAGE_LOG, toggleSetItem } from '@yandex-infracloud-ui/libs-next';

import { IListResult, ILogItem } from '../../models';
import {
   getDefaultLogFilters,
   GroupableLogItem,
   ILogFilters,
   ILogUrlParams,
   logFiltersToUrlParams,
   LogLayout,
} from './models';

// region Actions

export enum ActionType {
   AfterLoading,
   BeforeLoading,
   ExpandRow,
   SetFilters,
   SetLoading,
}

interface IActionAfterLoading {
   addToEnd: boolean;
   layout: LogLayout;
   resp: IListResult<ILogItem>;
   type: ActionType.AfterLoading;
}

interface IActionBeforeLoading {
   cursor: Date | undefined;
   type: ActionType.BeforeLoading;
}

interface IActionExpandRow {
   id: string;
   type: ActionType.ExpandRow;
}

interface IActionSetFilters {
   filters: ILogFilters;
   type: ActionType.SetFilters;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

type Action = IActionAfterLoading | IActionBeforeLoading | IActionExpandRow | IActionSetFilters | IActionSetLoading;

// endregion

// region State
export const initialState = {
   _allItems: [] as ILogItem[],
   expanded: new Set<string>(),
   filters: getDefaultLogFilters(),
   hasMore: false,
   isLoading: false,
   items: [] as ILogItem[],
   urlParams: {} as ILogUrlParams,
};

type State = Readonly<typeof initialState>;

export const initState = (state: State): State => ({
   ...state,
   urlParams: logFiltersToUrlParams(state.filters),
});
// endregion

// region Helpers

const _isDescendantOrder = (items: GroupableLogItem[]): boolean => {
   if (items.length < 2) {
      return false;
   }

   const first = items[0];
   const last = items[items.length - 1];

   return first.time > last.time;
};

/**
 * Что-то типа forEach, но с возможным указанием порядка сортировки
 */
const _iterate = <T>(reversed: boolean, list: T[], cb: (i: T) => void): void => {
   if (reversed) {
      for (let i = list.length - 1; i > 0; i -= 1) {
         cb(list[i]);
      }
   } else {
      list.forEach(cb);
   }
};

const _isSubItem = (item2: GroupableLogItem, item1: GroupableLogItem): boolean => {
   const time2 = item2.time;
   const time1 = item1.time;
   const statusTime1 = item1.status_time;

   return time2 > time1 && time2 < statusTime1;
};

export const _groupLogItems = (items: GroupableLogItem[]): GroupableLogItem[] => {
   const subItemsIds = new Set<string>();

   _iterate(_isDescendantOrder(items), items, item => {
      if (subItemsIds.has(item.id)) {
         return;
      }

      const subItems = items.filter(
         i =>
            i !== item &&
            i.host_inv === item.host_inv && // Группироваться могут только записи с одного хоста
            _isSubItem(i, item),
      );

      for (const si of subItems) {
         subItemsIds.add(si.id);
      }
      if (subItems.length) {
         item.subItems = subItems as ILogItem[];
      }
   });

   return items.filter(i => !subItemsIds.has(i.id));
};

// endregion

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.AfterLoading: {
         const items = action.resp.result;
         const allItems = action.addToEnd ? [...state._allItems, ...items] : items;

         return {
            ...state,
            _allItems: allItems,
            hasMore: items.length === PER_PAGE_LOG,
            isLoading: false,
            items:
               action.layout === 'host' || action.layout === 'single'
                  ? (_groupLogItems(allItems as GroupableLogItem[]) as ILogItem[])
                  : allItems,
         };
      }

      case ActionType.BeforeLoading: {
         return {
            ...state,
            expanded: action.cursor ? state.expanded : new Set(),
            isLoading: true,
            items: action.cursor ? state.items : [],
         };
      }

      case ActionType.ExpandRow: {
         return {
            ...state,
            expanded: toggleSetItem(state.expanded, action.id),
         };
      }

      case ActionType.SetFilters: {
         return isEqual(state.filters, action.filters) ? state : { ...state, filters: action.filters };
      }

      case ActionType.SetLoading: {
         return { ...state, isLoading: action.isLoading };
      }

      default: {
         return state;
      }
   }
};
