import { createIdGenerator, getIntRandom, measureDuration } from '../../utils/forTests';
import { _groupLogItems } from './AuditLog.state';
import { GroupableLogItem } from './models';

describe('AuditLog state: _groupLogItems', () => {
   it('should return empty if empty', () => {
      expect(_groupLogItems([])).toEqual([]);
   });

   it('should group normal', () => {
      const input: GroupableLogItem[] = [
         { id: '1', host_inv: 1, time: 1, status_time: 5 },
         { id: '2', host_inv: 1, time: 2, status_time: 3 },
      ];
      const output = [
         {
            host_inv: 1,
            id: '1',
            status_time: 5,
            subItems: [{ id: '2', host_inv: 1, time: 2, status_time: 3 }],
            time: 1,
         },
      ];

      expect(_groupLogItems(input)).toEqual(output);
   });

   it('should group normal (reversed)', () => {
      const input: GroupableLogItem[] = [
         { id: '2', host_inv: 1, time: 2, status_time: 3 },
         { id: '1', host_inv: 1, time: 1, status_time: 5 },
      ];
      const output = [
         {
            host_inv: 1,
            id: '1',
            status_time: 5,
            subItems: [{ id: '2', host_inv: 1, time: 2, status_time: 3 }],
            time: 1,
         },
      ];

      expect(_groupLogItems(input)).toEqual(output);
   });

   it('should group only for one level', () => {
      const input: GroupableLogItem[] = [
         { id: '1', host_inv: 1, time: 1, status_time: 10 },
         { id: '2', host_inv: 1, time: 2, status_time: 8 },
         { id: '3', host_inv: 1, time: 5, status_time: 6 },
      ];
      const output = [
         {
            host_inv: 1,
            id: '1',
            status_time: 10,
            subItems: [
               { id: '2', host_inv: 1, time: 2, status_time: 8 },
               { id: '3', host_inv: 1, time: 5, status_time: 6 },
            ],
            time: 1,
         },
      ];

      expect(_groupLogItems(input)).toEqual(output);
   });

   it('should group only for one level (reversed)', () => {
      const input: GroupableLogItem[] = [
         { id: '3', host_inv: 1, time: 5, status_time: 6 },
         { id: '2', host_inv: 1, time: 2, status_time: 8 },
         { id: '1', host_inv: 1, time: 1, status_time: 10 },
      ];
      const output = [
         {
            host_inv: 1,
            id: '1',
            status_time: 10,
            subItems: [
               { id: '3', host_inv: 1, time: 5, status_time: 6 },
               { id: '2', host_inv: 1, time: 2, status_time: 8 },
            ],
            time: 1,
         },
      ];

      expect(_groupLogItems(input)).toEqual(output);
   });

   it('should not group with different hosts', () => {
      const input: GroupableLogItem[] = [
         { id: '2', host_inv: 1, time: 2, status_time: 3 },
         { id: '1', host_inv: 2, time: 1, status_time: 5 },
      ];
      const output = [
         { host_inv: 1, id: '2', time: 2, status_time: 3 },
         { host_inv: 2, id: '1', status_time: 5, time: 1 },
      ];

      expect(_groupLogItems(input)).toEqual(output);
   });

   xit('should work quickly', () => {
      const COUNT = 10000; // 10 000^2 = 100 000 000 O(n2))
      const TIME_LIMIT = 2000;

      const startTime = Date.now();
      const nextId = createIdGenerator();

      function generateItem(): GroupableLogItem {
         const time = startTime + getIntRandom(0, 1000);

         return {
            host_inv: getIntRandom(0, 10000),
            id: nextId().toString(),
            status_time: time + getIntRandom(0, 10000),
            time,
         };
      }

      const input = Array(COUNT)
         .fill(null)
         .map(() => generateItem());
      const [duration, output] = measureDuration(() => _groupLogItems(input));

      console.table(output.filter(o => o.subItems));

      expect(input.length).toBeGreaterThan(output.length);
      expect(duration).toBeLessThan(TIME_LIMIT);
   });
});
