import { fromQuery, isEqual, NullableDate, toQuery, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer, useRef } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { auditLogApi } from '../../services';
import { LocationContext, SidebarDirection, SidebarToggleButton, useConstants } from '../../shared';
import { idSetToSet } from '../../state/commonModels';
import { projectsSlice } from '../../state/projects';
import { RootState } from '../../state/store';

import styles from './AuditLog.module.css';
import { ActionType, initialState, initState, reducer } from './AuditLog.state';
import { LogFilters } from './components/LogFilters';
import { LogList } from './components/LogList';
import {
   _detectLogType,
   ILogFilters,
   ILogUrlParams,
   logFiltersToUrlParams,
   LogLayout,
   LogType,
   urlParamsToLogFilters,
} from './models';

type Props = RouteComponentProps<ILogUrlParams> & {
   forceFilters?: Partial<ILogFilters>;
   id?: string;
   layout: LogLayout;
};

const getFiltersFromUrl = (search: string): ILogFilters => {
   const urlParams: ILogUrlParams = fromQuery(search);

   return urlParamsToLogFilters(urlParams);
};

export const AuditLog = React.memo((props: Props) => {
   const isSingle = props.layout === 'single' || props.layout === 'host-single';

   if (isSingle && !props.id) {
      throw new Error('AuditLog: id is required when layout is "single"');
   }

   //region hooks
   const hasForcedProject = props.forceFilters && props.forceFilters.project && props.forceFilters.project.size > 0;

   const firstRun = useRef(true);
   const previousRequest = useRef<Subscription>();
   const dismounted = useDismounted();

   const [state, dispatch] = useReducer(
      reducer,
      {
         ...initialState,
         filters: getFiltersFromUrl(props.location.search),
      },
      initState,
   );
   const { constants } = useConstants();

   const selectedProjects = useSelector((s: RootState) => s.projects.selectedIds);
   const reduxDispatch = useDispatch();
   //endregion

   //region helpers
   const loadItems = useCallback(
      (filters: ILogFilters, cursor?: Date) => {
         previousRequest.current = auditLogApi
            .getList({ ...filters, ...props.forceFilters }, cursor)
            .pipe(
               finalize(() => (previousRequest.current = undefined)),
               takeUntil(dismounted),
            )
            .subscribe(
               resp => {
                  dispatch({
                     addToEnd: Boolean(cursor),
                     layout: props.layout,
                     resp,
                     type: ActionType.AfterLoading,
                  });
               },
               resp => {
                  dispatch({ type: ActionType.SetLoading, isLoading: false });
                  toasts.apiError('Audit log loading', resp);
               },
            );
      },
      [dismounted, props.layout, props.forceFilters],
   );

   const load = useCallback(
      // Загрузка одной страницы с логом на основе фильтров
      (filters: ILogFilters, cursor?: Date) => {
         dispatch({ type: ActionType.BeforeLoading, cursor });

         if (previousRequest.current) {
            previousRequest.current.unsubscribe();
         }

         // Обычный аудитные лог
         if (!isSingle) {
            loadItems(filters, cursor);

            return;
         }

         // Аудитный лог отдельной записи
         previousRequest.current = auditLogApi
            .getById(props.id!)
            .pipe(
               finalize(() => (previousRequest.current = undefined)),
               takeUntil(dismounted),
            )
            .subscribe(
               item => {
                  dispatch({
                     addToEnd: false,
                     layout: props.layout,
                     resp: { result: [item] },
                     type: ActionType.AfterLoading,
                  });

                  dispatch({ type: ActionType.ExpandRow, id: item.id });

                  if (_detectLogType(item) !== LogType.Host) {
                     return;
                  }

                  const subItemFilters = {
                     ...filters,
                     fqdn: item.host_inv.toString(),
                     from: item.time as NullableDate,
                     to: item.status_time as NullableDate,
                  };

                  // Грузим дочерние записи
                  loadItems(subItemFilters, undefined);
               },
               resp => {
                  dispatch({ type: ActionType.SetLoading, isLoading: false });
                  toasts.apiError('Audit log loading', resp);
               },
            );
      },
      [dismounted, isSingle, loadItems, props.id, props.layout],
   );

   const updateUrl = useCallback(
      (filters: ILogFilters) => {
         const urlParams = logFiltersToUrlParams(filters);
         const search = toQuery(urlParams);
         if (search !== props.location.search) {
            props.history.push({ search });
         }
      },
      [props.history, props.location.search],
   );
   //endregion

   //region effects
   // Изменение фильтров при навигации в браузере
   useEffect(() => {
      const filters = getFiltersFromUrl(props.location.search);

      if (!hasForcedProject) {
         reduxDispatch(projectsSlice.actions.select(Array.from(filters.project)));
      }

      dispatch({ type: ActionType.SetFilters, filters });
      load(filters);
   }, [props.location.search, hasForcedProject, load, reduxDispatch]);

   // Подписка на изменение текущего проекта
   useEffect(() => {
      // Пропуск первого раза
      if (firstRun.current) {
         firstRun.current = false;

         return;
      }

      if (hasForcedProject) {
         return;
      }

      const project = idSetToSet(selectedProjects);
      if (isEqual(project, state.filters.project)) {
         return;
      }

      const filters = { ...state.filters, project };

      dispatch({ type: ActionType.SetFilters, filters });
      updateUrl(filters);
   }, [hasForcedProject, selectedProjects, state.filters, updateUrl]);
   //endregion

   //region handlers
   const onFiltersChange = useCallback(
      (e: SyntheticEvent | null, filters: ILogFilters) => {
         dispatch({ type: ActionType.SetFilters, filters });
         updateUrl(filters);
      },
      [updateUrl],
   );

   const loadMore = useCallback(() => {
      const lastItem = state.items[state.items.length - 1];
      if (lastItem) {
         load(state.filters, lastItem.time as Date);
      }
   }, [load, state.items, state.filters]);
   //endregion

   //region render
   return (
      <div className={styles.hostAuditLog}>
         {isSingle ? null : (
            <LogFilters
               value={state.filters}
               onChange={onFiltersChange}
               constants={constants}
               forceFilters={props.forceFilters}
            />
         )}

         <LocationContext.Provider value={props.location}>
            <LogList
               expanded={state.expanded}
               expand={id => dispatch({ type: ActionType.ExpandRow, id })}
               items={state.items}
               isLoading={state.isLoading}
               layout={props.layout}
            />
         </LocationContext.Provider>

         <div className={styles.tableFooter}>
            <SidebarToggleButton className={styles.sidebarToggleButton} direction={SidebarDirection.Expand} />

            {state.hasMore && !state.isLoading ? (
               <div className={styles.loadMore}>
                  <Button view={'action'} size={'m'} onClick={loadMore}>
                     Load more
                  </Button>
               </div>
            ) : null}
         </div>
      </div>
   );
   //endregion
});

AuditLog.displayName = 'AuditLog';
