import {
   arrayToSet,
   autobind,
   DateRangePicker,
   isEqual,
   NullableDate,
   toggleSetItem,
} from '@yandex-infracloud-ui/libs-next';
import { Button, YCSelect } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent } from 'react';
import { arrayToYCSelectItem } from 'utils/toLibs';

import { IConstants } from '../../../models';
import { FindUserButton, IssuerList, TextInputSearch, UserSelectMode } from '../../../shared';

import { getDefaultLogFilters, ILogFilters } from '../models';
import styles from './LogFilters.module.css';

interface IProps {
   constants: IConstants | null;
   forceFilters?: Partial<ILogFilters>;
   value: ILogFilters;

   onChange(e: SyntheticEvent | null, v: ILogFilters): void;
}

export class LogFilters extends React.PureComponent<IProps> {
   public render() {
      const { value, constants } = this.props;

      const logStatuses = constants ? arrayToYCSelectItem(constants.audit_log_statuses) : [];
      const logTypes = constants ? arrayToYCSelectItem(constants.audit_log_types) : [];
      const disabledFields = new Set(Object.keys(this.props.forceFilters || {}));

      return (
         <div className={styles.logFilters}>
            {/* FQDN */}
            <div>
               <TextInputSearch
                  view={'normal'}
                  size={'s'}
                  type={'text'}
                  name={'fqdn'}
                  placeholder={'FQDN, inv, UUID of hosts'}
                  value={(this.props.forceFilters && this.props.forceFilters.fqdn) || value.fqdn}
                  hasClear={true}
                  disabled={disabledFields.has('fqdn')}
                  onUpdate={this._onFQDNChange}
               />
            </div>

            {/* DATETIME */}
            <div>
               <DateRangePicker
                  // disabled={disabledFields.has('to') }|| disabledFields.has('from') TODO
                  to={value.to}
                  from={value.from}
                  onToChange={this._onToChange}
                  onFromChange={this._onFromChange}
                  withTime={true}
               />
            </div>

            {/* STATUS*/}
            <div>
               <YCSelect
                  onUpdate={this._onStatusChange}
                  disabled={disabledFields.has('status')}
                  type={'multiple'}
                  value={Array.from(value.status)}
                  items={logStatuses}
                  label={'Status:'}
                  placeholder={'all'}
                  stretched={false}
               />
            </div>

            {/* TYPES */}
            <div>
               <YCSelect
                  onUpdate={this._onTypeChange}
                  disabled={disabledFields.has('status')}
                  type={'multiple'}
                  value={Array.from(value.type)}
                  items={logTypes}
                  label={'Type:'}
                  placeholder={'all'}
               />
            </div>

            {/* ISSUERS */}
            <div className={styles.issuer}>
               Issuers:
               <FindUserButton
                  // disabled={disabledFields.has('issuer')} TODO
                  submitLabel={'Add'}
                  title={'Add issuer'}
                  onSelect={this._onSelectIssuer}
                  mode={UserSelectMode.Issuers}
                  selected={value.issuer}
               />
               <IssuerList editable={true} forceAvatars={true} value={value.issuer} onChange={this._onIssuerChange} />
            </div>

            <div>
               <Button view={'outlined'} size={'m'} onClick={this._onReset}>
                  Reset
               </Button>
            </div>
         </div>
      );
   }

   @autobind
   private _onFQDNChange(value: string | number) {
      this._tryChangeFilters(null, 'fqdn', String(value));
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onFromChange(e: null, v: NullableDate) {
      this._tryChangeFilters(e, 'from', v);
   }

   @autobind
   private _onIssuerChange(e: SyntheticEvent, issuer: Set<string>): void {
      this._tryChangeFilters(e, 'issuer', issuer);
   }

   @autobind
   private _onReset(e: React.MouseEvent<HTMLButtonElement>) {
      const defaultFilters = {
         ...getDefaultLogFilters(),
         project: this.props.value.project,
      };

      if (!isEqual(this.props.value, defaultFilters)) {
         this.props.onChange(e, defaultFilters);
      }
   }

   @autobind
   private _onSelectIssuer(e: SyntheticEvent, login: string | null) {
      if (login === null) {
         return;
      }

      const issuer = toggleSetItem(this.props.value.issuer, login);

      this._onIssuerChange(e, issuer);
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onStatusChange(v: string[]) {
      this._tryChangeFilters(null, 'status', arrayToSet(v));
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onToChange(e: null, v: NullableDate) {
      this._tryChangeFilters(e, 'to', v);
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onTypeChange(v: string[]) {
      this._tryChangeFilters(null, 'type', arrayToSet(v));
   }

   /**
    * Вынес повторяющийся код в отдельный метод
    */
   private _tryChangeFilters<K extends keyof ILogFilters>(
      e: SyntheticEvent | null,
      field: K,
      value: ILogFilters[K],
   ): void {
      const currentValue = this.props.value[field];

      if (!isEqual(value, currentValue)) {
         const newFilters = {
            ...this.props.value,
            [field]: value,
         };

         this.props.onChange(e, newFilters);
      }
   }
}
