import { autobind, ClampLines, classNames, CopyableText, isEmpty, Json } from '@yandex-infracloud-ui/libs-next';
import * as H from 'history';
import { Button } from '@yandex-data-ui/common';
import { bool, func, instanceOf, object, string } from 'prop-types';
import * as React from 'react';
import { Children, SyntheticEvent } from 'react';
import { Link } from 'react-router-dom';

import { ILogItem, ROUTE_LINKS } from '../../../models';
import { ClickOnlyUnmovedTrap, LocationContext, ProjectName, TextMessage, UserOrSystemName } from '../../../shared';

import { _detectLogType, getLogTypeHint, LogLayout, LogType } from '../models';
import styles from './LogItem.module.css';
import { TimePeriod } from './TimePeriod';
import { CloseIcon, FolderOpenIcon } from 'design/icons';

interface IProps {
   cls: string;
   expanded: Set<string>;
   layout: LogLayout;
   value: ILogItem;

   expand(id: string): void;

   onHover?(hovered: boolean): void;
}

interface IState {
   hovered: boolean;
}

export class LogItem extends React.PureComponent<IProps, IState> {
   public static defaultProps = {
      cls: '',
      expanded: false,
   };

   public static propTypes = {
      cls: string,
      expand: func.isRequired,
      expanded: instanceOf(Set).isRequired,
      subItemsVisible: bool,
      toggleSubItems: func,
      value: object.isRequired,
   };

   constructor(props: IProps) {
      super(props);

      this.state = {
         hovered: false,
      };
   }

   public render() {
      const li = this.props.value;
      const expanded = this.props.expanded.has(li.id);

      const isOneKindOfTarget = this.props.layout === 'host';
      const renderHostLink = this.props.layout !== 'host';
      const cells = (
         <>
            {isOneKindOfTarget ? null : this._renderLetterCell(li)}
            {this._renderTargetCell(li, renderHostLink)}
            {this._renderTypeCell(li)}
            {this._renderStatusCell(li)}
            {this._renderIssuerCell(li)}
            {this._renderTimeCell(li)}
            {this._renderInfoCell(li, expanded)}
         </>
      );

      return (
         <>
            {li.subItems &&
               li.subItems.map(si => (
                  <LogItem
                     key={si.id}
                     cls={classNames(styles.subItem, {
                        [styles._hoveredSubItem]: this.state.hovered,
                     })}
                     layout={this.props.layout}
                     expanded={this.props.expanded}
                     value={si}
                     expand={this.props.expand}
                     onHover={this._onSubItemHover}
                  />
               ))}

            <ClickOnlyUnmovedTrap
               as={'tr'}
               onClick={this._expandRow}
               onMouseEnter={this._hoverOn}
               onMouseLeave={this._hoverOff}
               className={classNames(
                  {
                     [styles._expanded]: expanded,
                     [styles._hoveredSubItem]: this.state.hovered,
                  },
                  this.props.cls,
                  styles.logItem,
               )}
            >
               {cells}
            </ClickOnlyUnmovedTrap>

            {expanded && (
               <tr className={styles._additional} onMouseEnter={this._hoverOn} onMouseLeave={this._hoverOff}>
                  <td colSpan={Children.count(cells.props.children)}>
                     {this._renderLinks(li)}
                     {li.payload ? <Json obj={li.payload} /> : <pre>No payload</pre>}
                  </td>
               </tr>
            )}
         </>
      );
   }

   @autobind
   private _expand() {
      this.props.expand(this.props.value.id);
   }

   @autobind
   private _expandRow(e: SyntheticEvent) {
      const target = e.target as EventTarget & Element;

      if (target.tagName !== 'TD' && target.tagName !== 'DIV') {
         return;
      }

      this._expand();
   }

   @autobind
   private _hoverOff() {
      const hovered = false;

      this.setState({ hovered });

      if (this.props.onHover) {
         this.props.onHover(hovered);
      }
   }

   @autobind
   private _hoverOn() {
      const hovered = true;

      this.setState({ hovered });

      if (this.props.onHover) {
         this.props.onHover(hovered);
      }
   }

   @autobind
   private _onSubItemHover(hovered: boolean) {
      this.setState({ hovered });
   }

   private _renderInfoCell(item: ILogItem, expanded: boolean) {
      return (
         <td className={styles.infoColumn}>
            <Button
               view={'clear'}
               size={'s'}
               className={classNames(
                  {
                     [styles.closeButtonHidden]: !expanded,
                  },
                  styles.closeButton,
               )}
               onClick={this._expand}
            >
               <CloseIcon size={'lg'} />
            </Button>

            <ClampLines count={2} enabled={!expanded}>
               {item.error ? (
                  <TextMessage
                     cls={styles.message}
                     hasSeparator={false}
                     isError={true}
                     limitHeight={false}
                     text={item.error}
                  />
               ) : null}
               {item.reason ? (
                  <TextMessage cls={styles.message} hasSeparator={false} limitHeight={false} text={item.reason} />
               ) : null}
            </ClampLines>
         </td>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderIssuerCell(li: ILogItem) {
      return (
         <td className={styles.nowrap}>
            <UserOrSystemName value={li.issuer} />
         </td>
      );
   }

   private _renderLetterCell(li: ILogItem) {
      const type = _detectLogType(li);

      return <td title={getLogTypeHint(type)}>{type}</td>;
   }

   private _renderLinkToScenario(li: ILogItem, location: H.Location | null) {
      return li.scenario_id ? (
         <>
            <br />
            <Link
               to={{
                  pathname: ROUTE_LINKS.scenario(li.scenario_id),
                  state: { prev: location },
               }}
            >
               Scenario #{li.scenario_id}
            </Link>
         </>
      ) : null;
   }

   // noinspection JSMethodCanBeStatic
   private _renderLinks(li: ILogItem) {
      const itemUrl = `${window.location.origin}${ROUTE_LINKS.auditLogItem(li.id)}`;
      const operationLogUri =
         this.props.layout === 'host' || this.props.layout === 'host-single'
            ? ROUTE_LINKS.hostOperationLog(li.host_name, { audit_log_id: li.id })
            : ROUTE_LINKS.operationLog({ audit_log_id: li.id });

      return (
         <div className={styles.links}>
            <span>
               <Link to={ROUTE_LINKS.auditLogItem(li.id)}>Link to this item</Link>

               <CopyableText text={itemUrl} className={styles.copyIcon} />
            </span>

            <span>
               <Link to={operationLogUri}>Operation log</Link>

               <CopyableText text={`${window.location.origin}${operationLogUri}`} className={styles.copyIcon} />
            </span>
         </div>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderStatusCell(li: ILogItem) {
      return <td title={li.status}>{li.status}</td>;
   }

   private _renderTargetCell(li: ILogItem, renderHostLink: boolean) {
      const type = _detectLogType(li);

      const columnStyle = classNames(styles.targetColumn, styles.isGrouped);

      return (
         <td>
            <LocationContext.Consumer>
               {location => (
                  <div className={columnStyle}>
                     <div>
                        {/* HOST */}
                        {type === LogType.Host ? (
                           <>
                              {renderHostLink ? (
                                 <Link
                                    to={{
                                       pathname: ROUTE_LINKS.hostAuditLog(li.host_name),
                                       state: { prev: location },
                                    }}
                                 >
                                    {li.host_name}
                                 </Link>
                              ) : (
                                 li.host_name
                              )}

                              {isEmpty(li.subItems) ? null : (
                                 <Button view={'clear'} size={'s'} className={styles.groupButton}>
                                    <FolderOpenIcon />
                                 </Button>
                              )}

                              <div className={styles.secondString}>
                                 Inv: {li.host_inv}
                                 {li.project ? (
                                    <>
                                       &nbsp;
                                       <ProjectName id={li.project} />
                                    </>
                                 ) : null}
                                 {this._renderLinkToScenario(li, location)}
                              </div>
                           </>
                        ) : null}

                        {/* PROJECT */}
                        {type === LogType.Project ? (
                           <>
                              <Link
                                 to={{
                                    pathname: ROUTE_LINKS.project(li.project),
                                    state: { prev: location },
                                 }}
                              >
                                 {li.project}
                              </Link>

                              <div className={styles.secondString}>
                                 <ProjectName id={li.project} />
                                 {this._renderLinkToScenario(li, location)}
                              </div>
                           </>
                        ) : null}

                        {/* AUTOMATION */}
                        {type === LogType.Automation ? (
                           <>
                              <Link
                                 to={{
                                    pathname: ROUTE_LINKS.automationPlot(li.automation_plot!),
                                    state: { prev: location },
                                 }}
                              >
                                 {li.automation_plot}
                              </Link>

                              <div className={styles.secondString}>
                                 automation plot
                                 {this._renderLinkToScenario(li, location)}
                              </div>
                           </>
                        ) : null}

                        {/* SCENARIO */}
                        {type === LogType.Scenario ? (
                           <>
                              <Link
                                 to={{
                                    pathname: ROUTE_LINKS.scenario(li.scenario_id!),
                                    state: { prev: location },
                                 }}
                              >
                                 {li.scenario_id}
                              </Link>

                              <div className={styles.secondString}>scenario</div>
                           </>
                        ) : null}

                        {/* WALLE (Global) */}
                        {type === LogType.WallE ? <>&mdash;</> : null}
                     </div>
                  </div>
               )}
            </LocationContext.Consumer>
         </td>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderTimeCell(li: ILogItem) {
      return (
         <td>
            <div className={styles.dateColumn}>
               <span className={styles.date}>
                  <TimePeriod start={li.time as Date} end={li.status_time as Date} />
               </span>
            </div>
         </td>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderTypeCell(li: ILogItem) {
      return (
         <td title={li.type} className={styles.typeColumn}>
            {li.type}
         </td>
      );
   }
}
