import { NullableDate, queryValueToSet, setToQueryValue } from '@yandex-infracloud-ui/libs-next';
import { ILogItem } from '../../models';
import { dateToUrl, urlToDate } from './dateInUrl';

export interface ILogFilters {
   id: Set<string>;
   fqdn: string;
   from: NullableDate;
   issuer: Set<string>;
   // в идеале тут Set<number>, то слишком много придется подстраивать,
   // string подойдет без побочных эффектов
   preorder: Set<string>;
   project: Set<string>;
   scenario_id: Set<string>;
   status: Set<string>;
   to: NullableDate;
   type: Set<string>;
}

export interface ILogUrlParams {
   fqdn?: string;
   from?: string;
   id?: string;
   issuer?: string;
   preorder?: string;
   project?: string;
   scenario_id?: string;
   status?: string;
   to?: string;
   type?: string;
}

export function logFiltersToUrlParams(filters: ILogFilters): ILogUrlParams {
   return {
      fqdn: filters.fqdn,
      from: filters.from ? dateToUrl(filters.from) : undefined,
      issuer: setToQueryValue(filters.issuer),
      preorder: setToQueryValue(filters.preorder),
      project: setToQueryValue(filters.project),
      scenario_id: setToQueryValue(filters.scenario_id),
      status: setToQueryValue(filters.status),
      to: filters.to ? dateToUrl(filters.to) : undefined,
      type: setToQueryValue(filters.type),
   };
}

export function urlParamsToLogFilters(query: ILogUrlParams): ILogFilters {
   return {
      id: queryValueToSet(query.id),
      fqdn: query.fqdn || '',
      from: query.from ? urlToDate(query.from) : null,
      issuer: queryValueToSet(query.issuer),
      preorder: queryValueToSet(query.preorder),
      project: queryValueToSet(query.project),
      scenario_id: queryValueToSet(query.scenario_id),
      status: queryValueToSet(query.status),
      to: query.to ? urlToDate(query.to, true) : null,
      type: queryValueToSet(query.type),
   };
}

export function getDefaultLogFilters(): ILogFilters {
   return {
      id: new Set(),
      fqdn: '',
      from: null,
      issuer: new Set(),
      preorder: new Set(),
      project: new Set(),
      scenario_id: new Set(),
      status: new Set(),
      to: null,
      type: new Set(),
   };
}

export type GroupableLogItem = Pick<ILogItem, 'id' | 'host_inv' | 'time' | 'status_time' | 'subItems'>;

export type LogLayout = 'host' | 'all' | 'single' | 'host-single';

export enum LogType {
   Host = 'H',
   Project = 'P',
   Automation = 'A',
   Scenario = 'S',
   WallE = 'W',
}

/**
 * Из обсуждения в макете:
 * сейчас единственный способ отличать:
 * Если указаны параметры хоста (инвентарный номер и опционально имя), то эта запись привязана к хосту.
 * Если указан id проекта, то запись привязана к проекту.
 * Если указан id схемы проверок, то запись привязана к схеме проверок
 */
export const _detectLogType = (li: ILogItem): LogType => {
   if (li.host_inv) {
      return LogType.Host;
   }

   if (li.project) {
      return LogType.Project;
   }

   if (li.automation_plot) {
      return LogType.Automation;
   }

   if (li.scenario_id) {
      return LogType.Scenario;
   }

   return LogType.WallE;
};

export function getLogTypeHint(type: LogType): string {
   for (const name in LogType) {
      // noinspection JSUnfilteredForInLoop
      if (LogType[name] === type) {
         // noinspection JSUnfilteredForInLoop
         return name;
      }
   }

   return 'Unknown';
}
