import { Suggest, Switch } from '@yandex-data-ui/common';
import React, { useCallback, useState, useEffect } from 'react';
import { SyntheticEvent } from 'react';

import { IBotProject } from '../../models';
import { botProjectApi } from './botProjectApi';
import { BotProjectItem } from './BotProjectItem';

import styles from './BotProjectInput.module.css';
import { isEmpty } from 'lodash';

interface IProps {
   cls?: string;
   disabled?: boolean;
   initialQuery?: string;
   name?: string;

   onSelect(e: SyntheticEvent | null, projectId: number | string | null): void;

   onBlur?(): void;
}

export const BotProjectInput: React.FC<IProps> = props => {
   const { onSelect, initialQuery, disabled, cls, name } = props;
   const [query, setQuery] = useState(initialQuery);

   const [showAll, setShowAll] = useState(false);

   const handleClick = useCallback(
      (v: IBotProject) => {
         setQuery(() => String(v.project_id));
         onSelect(null, v.project_id ?? null);
      },
      [onSelect],
   );

   const onChange = useCallback(
      (v: string) => {
         setQuery(() => v);
         onSelect(null, !isEmpty(v) ? (isNaN(Number(v)) ? v : Number(v)) : null);
      },
      [onSelect],
   );

   const handleGetItems = useCallback(
      async (q): Promise<IBotProject[]> => {
         if (!q) return [];

         return botProjectApi.getSuggestions(q, showAll).toPromise();
      },
      [showAll],
   );

   useEffect(() => {
      setQuery(() => initialQuery);
   }, [initialQuery]);

   return (
      <div className={styles.root}>
         <Suggest
            className={cls}
            // необходимо для триггера getItems
            key={name + String(showAll)}
            getItems={handleGetItems}
            text={query}
            getItemsOnMount={true}
            onUpdate={onChange}
            onBlur={props.onBlur}
            onItemClick={handleClick}
            onInputEnterKeyDown={onChange}
            applicableInputValue={true}
            disabled={disabled}
            placeholder={'Enter BOT Project ID'}
            renderItem={item => <BotProjectItem item={item} />}
            debounce={500}
            syncPopupOnResize={true}
         />

         <Switch
            className={styles.switch}
            disabled={disabled}
            checked={showAll}
            onUpdate={setShowAll}
            content={'Show deleted'}
            size={'m'}
         />
      </div>
   );
};

BotProjectInput.displayName = 'BotProjectInput';
