import { sortHandler } from '@yandex-infracloud-ui/libs';

import { IStaffPerson, NotificationRecipients } from '../../models';

import { ILevel, IRow, rowsToValue, toggleLevel, valueToRows } from './models';

export enum ActionType {
   AddRow,
   RemoveRow,
   SetLoading,
   SetNames,
   SetRows,
   SetShowAddUser,
   Toggle,
}

interface IActionAddRow {
   row: IRow;
   type: ActionType.AddRow;
}

interface IActionRemoveRow {
   row: IRow;
   type: ActionType.RemoveRow;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

interface IActionSetNames {
   persons: IStaffPerson[];
   type: ActionType.SetNames;
}

interface IActionSetRows {
   rows: IRow[];
   type: ActionType.SetRows;
}

interface IActionSetShowAddUser {
   showAddUser: boolean;
   type: ActionType.SetShowAddUser;
}

interface IActionToggle {
   level: ILevel;
   row: IRow;
   type: ActionType.Toggle;
}

type Action =
   | IActionAddRow
   | IActionRemoveRow
   | IActionSetLoading
   | IActionSetNames
   | IActionSetRows
   | IActionSetShowAddUser
   | IActionToggle;

export const initialState = {
   isLoading: true,
   rows: [] as IRow[],
   showAddUser: false,
   value: {} as NotificationRecipients,
};

type State = Readonly<typeof initialState>;

export const initState = (state: State): State =>
   reducer(state, {
      rows: valueToRows(state.value!),
      type: ActionType.SetRows,
   });

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.AddRow: {
         const rows = [...state.rows, action.row];

         return reducer(state, { type: ActionType.SetRows, rows });
      }

      case ActionType.RemoveRow: {
         const rows = state.rows.filter(row => row !== action.row);

         return reducer(state, { type: ActionType.SetRows, rows });
      }

      case ActionType.SetLoading: {
         return { ...state, isLoading: action.isLoading };
      }

      case ActionType.SetNames: {
         const rows = state.rows.map(row => {
            // Уже есть имя
            if (row.name) {
               return row;
            }

            const person = action.persons.find(p => p.email === row.email);

            return {
               ...row,
               name: person ? person.name : row.email,
            };
         });

         return reducer({ ...state, isLoading: false }, { type: ActionType.SetRows, rows });
      }

      case ActionType.SetRows: {
         const sorted = [...action.rows];
         sorted.sort((a, b) => sortHandler(a.email, b.email));

         return { ...state, rows: sorted, value: rowsToValue(sorted) };
      }

      case ActionType.SetShowAddUser: {
         return { ...state, showAddUser: action.showAddUser };
      }

      case ActionType.Toggle: {
         const rows = state.rows.map(row => {
            if (row !== action.row) {
               return row;
            }

            return {
               ...row,
               levels: toggleLevel(row.levels, action.level),
            };
         });

         return reducer(state, { type: ActionType.SetRows, rows });
      }

      default: {
         console.warn('Unknown action', action);

         return state;
      }
   }
};
