import { Suggest } from '@yandex-data-ui/common';
import {
   EMAIL_HOST,
   IStaffPerson,
   IValueProps,
   SuggestLayer,
   toasts,
   useDismounted,
   getStrategy,
} from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useReducer, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { NotificationRecipients } from '../../models';
import { staffApi } from '../../services';
import { DevJson } from '../../shared';

import { EmailNotificationRow } from './EmailNotificationRow';
import styles from './EmailNotifications.module.css';
import { ActionType, initialState, initState, reducer } from './EmailNotifications.state';
import { getEmptyRow, ILevel, IRow, SEVERITY_LEVELS } from './models';

interface Props extends IValueProps<NotificationRecipients> {
   readonly?: boolean;
}

const strategy = getStrategy(SuggestLayer.People);

export const EmailNotifications: React.FC<Props> = React.memo(({ readonly = false, value, onChange }) => {
   // hooks
   const [query, setQuery] = useState('');
   const dismounted = useDismounted();
   const [{ rows, isLoading, value: stateValue }, dispatch] = useReducer(
      reducer,
      { ...initialState, value },
      initState,
   );

   // Загрузка имён пользователей для тех, у кого их нет еще
   useEffect(() => {
      const emailToLoad = rows.filter(r => !r.name).map(r => r.email);

      if (emailToLoad.length === 0) {
         return;
      }

      dispatch({ type: ActionType.SetLoading, isLoading: true });

      staffApi
         .getRichPersonsByEmails(emailToLoad)
         .pipe(takeUntil(dismounted))
         .subscribe(
            persons => dispatch({ type: ActionType.SetNames, persons }),
            resp => {
               dispatch({ type: ActionType.SetLoading, isLoading: false });
               toasts.apiError('Staff loading', resp);
            },
         );
   }, [dismounted, rows]);

   // Обновление внешней модели при обновлении внутренней
   useEffect(() => onChange(null, stateValue), [onChange, stateValue]);

   // handlers
   const toggle = useCallback((r: IRow, l: ILevel) => dispatch({ type: ActionType.Toggle, row: r, level: l }), []);

   const remove = useCallback((r: IRow) => dispatch({ type: ActionType.RemoveRow, row: r }), []);

   const addNewUser = useCallback((person: IStaffPerson | null) => {
      if (!person) {
         return;
      }

      dispatch({
         row: getEmptyRow(`${person.login}${EMAIL_HOST}`),
         type: ActionType.AddRow,
      });
   }, []);

   const handleGetItems = useCallback(async (q): Promise<IStaffPerson[] | any> => {
      if (!q) return [];

      return strategy
         .load(q)
         .toPromise()
         .then((resp: any) => resp.slice(0, 10));
   }, []);

   // render
   return (
      <>
         {readonly ? null : (
            <Suggest
               getItems={handleGetItems}
               className={styles.userSelect}
               text={query}
               getItemsOnMount={true}
               onUpdate={setQuery}
               // onInputEnterKeyDown={onChange}
               onItemClick={addNewUser}
               applicableInputValue={true}
               placeholder={'Login or group'}
               renderItem={item => strategy.renderItem(item as any, query)}
               debounce={500}
               syncPopupOnResize={false}
            />
         )}

         {rows.length > 0 ? (
            <table className={styles.table}>
               <thead className={styles.head}>
                  <tr className={styles.headRow}>
                     <th>Email</th>
                     {SEVERITY_LEVELS.map(c => (
                        <th key={c.id}>{c.title}</th>
                     ))}
                     <th />
                  </tr>
               </thead>

               <tbody>
                  {rows.map(row => (
                     <EmailNotificationRow
                        readonly={readonly}
                        key={row.email}
                        isLoading={isLoading}
                        row={row}
                        onChange={toggle}
                        onRemove={remove}
                     />
                  ))}
               </tbody>
            </table>
         ) : (
            <label className={styles.noEmails}>There are not added emails for notifications</label>
         )}

         <DevJson>{value}</DevJson>
      </>
   );
});

EmailNotifications.displayName = 'EmailNotifications';
