import { createEmptyNotificationRecipients, NotificationRecipients, NotificationSeverity } from '../../models';

export interface ILevel {
   id: NotificationSeverity;
   level?: number;
   title: string;
}

export const SEVERITY_LEVELS: ILevel[] = [
   { title: 'BOT', id: NotificationSeverity.Bot },
   { title: 'Audit', id: NotificationSeverity.Audit, level: 0 },
   { title: 'Info', id: NotificationSeverity.Info, level: 1 },
   { title: 'Warn', id: NotificationSeverity.Warning, level: 2 },
   { title: 'Error', id: NotificationSeverity.Error, level: 3 },
   { title: 'Crit', id: NotificationSeverity.Critical, level: 4 },
];

type RowLevels = {
   [key in NotificationSeverity]?: boolean;
};

export interface IRow {
   email: string;
   levels: RowLevels;
   name?: string;
}

export function isImplicitChecked(levels: RowLevels, level?: number): boolean {
   // Исключение для BOT
   if (level === undefined) {
      return false;
   }

   for (const sLevel of SEVERITY_LEVELS) {
      if (sLevel.level === undefined) {
         continue;
      }
      if (levels[sLevel.id] && sLevel.level < level) {
         return true;
      }
   }

   return false;
}

export function getEmptyRow(email: string): IRow {
   // false для всех уровней по-умолчанию
   const levels = SEVERITY_LEVELS.reduce((acc, { id }) => {
      acc[id] = false;

      return acc;
   }, {} as RowLevels);

   return { email, levels };
}

export function toggleLevel(levels: RowLevels, level: ILevel): RowLevels {
   const result = getEmptyRow('').levels; // все false

   // Уровень не задан, переключаем
   if (level.level === undefined) {
      result[level.id] = !levels[level.id];

      // А остальное копируем без изменений
      for (const sLevel of SEVERITY_LEVELS) {
         if (sLevel.id !== level.id) {
            result[sLevel.id] = levels[sLevel.id];
         }
      }
   } else {
      // Если был выключен - включаем (остальные уже false)
      if (!levels[level.id]) {
         result[level.id] = true;
      }

      // Копируем те, что без весов
      for (const sLevel of SEVERITY_LEVELS) {
         if (sLevel.level === undefined) {
            result[sLevel.id] = levels[sLevel.id];
         }
      }
   }

   return result;
}

export function removeRecipients(
   recipients: NotificationRecipients,
   emailsToDelete: Set<string>,
): NotificationRecipients {
   const result: NotificationRecipients = { ...recipients };

   Object.entries(recipients).forEach(([severity, emails]) => {
      result[severity] = emails.filter(email => !emailsToDelete.has(email));
   });

   return result;
}

export function rowsToValue(rows: IRow[]): NotificationRecipients {
   const result = createEmptyNotificationRecipients();

   for (const { id } of SEVERITY_LEVELS) {
      for (const row of rows) {
         if (row.levels[id]) {
            if (!(id in result)) {
               result[id] = [];
            }
            result[id]!.push(row.email);
         }
      }
   }

   return result;
}

export function valueToRows(value: NotificationRecipients): IRow[] {
   const rowMap = new Map<string, IRow>();

   Object.keys(value).forEach(level => {
      const emails = value[level];

      for (const email of emails) {
         if (!rowMap.has(email)) {
            rowMap.set(email, getEmptyRow(email));
         }
         const row = rowMap.get(email)!;
         row.levels[level] = true;
      }
   });

   return Array.from(rowMap.values());
}
