import { isEqual, PER_PAGE_LOG, toggleSetItem } from '@yandex-infracloud-ui/libs-next';

import { IListResult, IOperationLogItem } from '../../models';
import {
   getDefaultOperationLogFilters,
   IOperationLogFilters,
   IOperationLogUrlParams,
   operationLogFiltersToUrlParams,
   OperationLogLayout,
} from './models';

// region Actions

export enum ActionType {
   AfterLoading,
   BeforeLoading,
   ExpandRow,
   SetFilters,
   SetLoading,
}

interface IActionAfterLoading {
   addToEnd: boolean;
   layout: OperationLogLayout;
   resp: IListResult<IOperationLogItem>;
   type: ActionType.AfterLoading;
}

interface IActionBeforeLoading {
   cursor: Date | undefined;
   type: ActionType.BeforeLoading;
}

interface IActionExpandRow {
   id: string;
   type: ActionType.ExpandRow;
}

interface IActionSetFilters {
   filters: IOperationLogFilters;
   type: ActionType.SetFilters;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

type Action = IActionAfterLoading | IActionBeforeLoading | IActionExpandRow | IActionSetFilters | IActionSetLoading;

// endregion

// region State
export const initialState = {
   _allItems: [] as IOperationLogItem[],
   expanded: new Set<string>(),
   filters: getDefaultOperationLogFilters(),
   hasMore: false,
   isLoading: false,
   items: [] as IOperationLogItem[],
   urlParams: {} as IOperationLogUrlParams,
};

type State = Readonly<typeof initialState>;

export const initState = (state: State): State => ({
   ...state,
   urlParams: operationLogFiltersToUrlParams(state.filters),
});
// endregion

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.AfterLoading: {
         const items = action.resp.result;
         const allItems = action.addToEnd ? [...state._allItems, ...items] : items;
         const itemsGroups = allItems.reduce((acc, item, index) => ({ ...acc, [item.audit_log_id]: index }), {});

         return {
            ...state,
            _allItems: allItems,
            hasMore: items.length === PER_PAGE_LOG,
            isLoading: false,
            items: allItems
               // группируем записи с одинаковым audit_log_id
               .sort((a1, a2) =>
                  a2.audit_log_id === a1.audit_log_id ? 0 : itemsGroups[a1.audit_log_id] - itemsGroups[a2.audit_log_id],
               )
               .map((item, index, array) => {
                  const nextItem = array[index + 1];

                  // если несколько operation log подряд относятся к одному аудитному логу,
                  // то выделяем их в группу, родитель при этом будет снизу
                  if (nextItem?.audit_log_id === item.audit_log_id) {
                     return { ...item, isSubItem: true };
                  }

                  return item;
               }),
         };
      }

      case ActionType.BeforeLoading: {
         return {
            ...state,
            expanded: action.cursor ? state.expanded : new Set(),
            isLoading: true,
            items: action.cursor ? state.items : [],
         };
      }

      case ActionType.ExpandRow: {
         return {
            ...state,
            expanded: toggleSetItem(state.expanded, action.id),
         };
      }

      case ActionType.SetFilters: {
         return isEqual(state.filters, action.filters) ? state : { ...state, filters: action.filters };
      }

      case ActionType.SetLoading: {
         return { ...state, isLoading: action.isLoading };
      }

      default: {
         return state;
      }
   }
};
