import { fromQuery, isEqual, toQuery, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer, useRef } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { operationLogApi } from '../../services';
import { LocationContext, SidebarDirection, SidebarToggleButton, useConstants } from '../../shared';
import { idSetToSet } from '../../state/commonModels';
import { projectsSlice } from '../../state/projects';
import { RootState } from '../../state/store';
import {
   IOperationLogFilters,
   IOperationLogUrlParams,
   operationLogFiltersToUrlParams,
   OperationLogLayout,
   urlParamsToOperationLogFilters,
} from './models';

import styles from '../AuditLog/AuditLog.module.css';
import { OperationLogFilters } from './components/OperationLogFilters';
import { OperationLogList } from './components/OperationLogList';
import { ActionType, initialState, initState, reducer } from './OperationLog.state';

type Props = RouteComponentProps<IOperationLogUrlParams> & {
   forceFilters?: Partial<IOperationLogFilters>;
   id?: string;
   layout: OperationLogLayout;
};

const getFiltersFromUrl = (search: string): IOperationLogFilters => {
   const urlParams: IOperationLogUrlParams = fromQuery(search);

   return urlParamsToOperationLogFilters(urlParams);
};

export const OperationLog = React.memo((props: Props) => {
   const isSingle = props.layout === 'single';

   //region hooks
   const hasForcedProject = props.forceFilters && props.forceFilters.project && props.forceFilters.project.size > 0;

   const firstRun = useRef(true);
   const previousRequest = useRef<Subscription>();
   const dismounted = useDismounted();

   const [state, dispatch] = useReducer(
      reducer,
      {
         ...initialState,
         filters: getFiltersFromUrl(props.location.search),
      },
      initState,
   );
   const { constants } = useConstants();

   const selectedProjects = useSelector((s: RootState) => s.projects.selectedIds);
   const reduxDispatch = useDispatch();
   //endregion

   //region helpers
   const loadItems = useCallback(
      (filters: IOperationLogFilters, cursor?: Date) => {
         previousRequest.current = operationLogApi
            .getList({ ...filters, ...props.forceFilters }, cursor)
            .pipe(
               finalize(() => (previousRequest.current = undefined)),
               takeUntil(dismounted),
            )
            .subscribe(
               resp => {
                  dispatch({
                     addToEnd: Boolean(cursor),
                     layout: props.layout,
                     resp,
                     type: ActionType.AfterLoading,
                  });
               },
               resp => {
                  dispatch({ type: ActionType.SetLoading, isLoading: false });
                  toasts.apiError('Operation log loading', resp);
               },
            );
      },
      [dismounted, props.layout, props.forceFilters],
   );

   const load = useCallback(
      // Загрузка одной страницы с логом на основе фильтров
      (filters: IOperationLogFilters, cursor?: Date) => {
         dispatch({ type: ActionType.BeforeLoading, cursor });

         if (previousRequest.current) {
            previousRequest.current.unsubscribe();
         }

         loadItems(filters, cursor);
      },
      [loadItems],
   );

   const updateUrl = useCallback(
      (filters: IOperationLogFilters) => {
         const urlParams = operationLogFiltersToUrlParams(filters);
         const search = toQuery(urlParams);
         if (search !== props.location.search) {
            props.history.push({ search });
         }
      },
      [props.history, props.location.search],
   );
   //endregion

   //region effects
   // Изменение фильтров при навигации в браузере
   useEffect(() => {
      const filters = getFiltersFromUrl(props.location.search);

      if (!hasForcedProject) {
         reduxDispatch(projectsSlice.actions.select(Array.from(filters.project)));
      }

      dispatch({ type: ActionType.SetFilters, filters });
      load(filters);
   }, [props.location.search, hasForcedProject, load, reduxDispatch]);

   // Подписка на изменение текущего проекта
   useEffect(() => {
      // Пропуск первого раза
      if (firstRun.current) {
         firstRun.current = false;

         return;
      }

      if (hasForcedProject) {
         return;
      }

      const project = idSetToSet(selectedProjects);
      if (isEqual(project, state.filters.project)) {
         return;
      }

      const filters = { ...state.filters, project };

      dispatch({ type: ActionType.SetFilters, filters });
      updateUrl(filters);
   }, [hasForcedProject, selectedProjects, state.filters, updateUrl]);
   //endregion

   //region handlers
   const onFiltersChange = useCallback(
      (e: SyntheticEvent | null, filters: IOperationLogFilters) => {
         dispatch({ type: ActionType.SetFilters, filters });
         updateUrl(filters);
      },
      [updateUrl],
   );

   const loadMore = useCallback(() => {
      const lastItem = state.items[state.items.length - 1];
      if (lastItem) {
         load(state.filters, lastItem.time as Date);
      }
   }, [load, state.items, state.filters]);
   //endregion

   //region render
   return (
      <div className={styles.hostAuditLog}>
         {isSingle ? null : (
            <OperationLogFilters
               value={state.filters}
               onChange={onFiltersChange}
               constants={constants}
               forceFilters={props.forceFilters}
            />
         )}

         <LocationContext.Provider value={props.location}>
            <OperationLogList
               expanded={state.expanded}
               expand={id => dispatch({ type: ActionType.ExpandRow, id })}
               items={state.items}
               isLoading={state.isLoading}
               layout={props.layout}
            />
         </LocationContext.Provider>

         <div className={styles.tableFooter}>
            <SidebarToggleButton className={styles.sidebarToggleButton} direction={SidebarDirection.Expand} />

            {state.hasMore && !state.isLoading ? (
               <div className={styles.loadMore}>
                  <Button view={'action'} size={'m'} onClick={loadMore}>
                     Load more
                  </Button>
               </div>
            ) : null}
         </div>
      </div>
   );
   //endregion
});

OperationLog.displayName = 'OperationLog';
