import { Button, Label, YCSelect } from '@yandex-data-ui/common';
import { arrayToSet, autobind, DateRangePicker, isEqual, NullableDate } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { SyntheticEvent } from 'react';

import { arrayToYCSelectItem } from 'utils/toLibs';

import { IConstants } from '../../../models';
import { TextInputSearch } from '../../../shared';

import { getDefaultOperationLogFilters, IOperationLogFilters } from '../models';
import styles from './OperationLogFilters.module.css';

interface IProps {
   constants: IConstants | null;
   forceFilters?: Partial<IOperationLogFilters>;
   value: IOperationLogFilters;

   onChange(e: SyntheticEvent | null, v: IOperationLogFilters): void;
}

export class OperationLogFilters extends React.PureComponent<IProps> {
   public render() {
      const { value, constants } = this.props;

      const logTypes = constants ? arrayToYCSelectItem(constants.operation_log_types) : [];
      const disabledFields = new Set(Object.keys(this.props.forceFilters || {}));

      return (
         <div className={styles.logFilters}>
            {/* FQDN */}
            <div>
               <TextInputSearch
                  view={'normal'}
                  type={'text'}
                  name={'fqdn'}
                  disabled={disabledFields.has('fqdn')}
                  placeholder={'FQDN, inv, UUID of hosts'}
                  value={(this.props.forceFilters && this.props.forceFilters.fqdn) || value.fqdn}
                  hasClear={true}
                  onUpdate={this._onFQDNChange}
               />
            </div>

            {/* DATETIME */}
            <div>
               <DateRangePicker
                  // disabled={disabledFields.has('to') }|| disabledFields.has('from') TODO
                  to={value.to}
                  from={value.from}
                  onToChange={this._onToChange}
                  onFromChange={this._onFromChange}
                  withTime={true}
               />
            </div>

            {/* TYPES */}
            <YCSelect
               onUpdate={this._onTypeChange}
               disabled={disabledFields.has('type')}
               type={'multiple'}
               value={Array.from(value.type)}
               items={logTypes}
               label={'Status:'}
               placeholder={'all'}
               stretched={false}
            />

            {/* AUDIT_LOG_ID */}
            {value.audit_log_id.size ? (
               <div className={styles.auditLog}>
                  Audit log:{' '}
                  <Label size={'s'} type={'close'} onClose={this._onAuditLogIdReset}>
                     {Array.from(value.audit_log_id).join(', ')}
                  </Label>
               </div>
            ) : null}

            <div className={styles.reset}>
               <Button view={'outlined'} size={'m'} onClick={this._onReset}>
                  Reset
               </Button>
            </div>
         </div>
      );
   }

   @autobind
   private _onAuditLogIdReset() {
      this._tryChangeFilters(null, 'audit_log_id', new Set());
   }

   @autobind
   private _onFQDNChange(value: string | number) {
      this._tryChangeFilters(null, 'fqdn', String(value));
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onFromChange(e: null, v: NullableDate) {
      this._tryChangeFilters(e, 'from', v);
   }

   @autobind
   private _onReset(e: React.MouseEvent<HTMLButtonElement>) {
      const defaultFilters = {
         ...getDefaultOperationLogFilters(),
         project: this.props.value.project,
      };

      if (!isEqual(this.props.value, defaultFilters)) {
         this.props.onChange(e, defaultFilters);
      }
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onToChange(e: null, v: NullableDate) {
      this._tryChangeFilters(e, 'to', v);
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onTypeChange(v: string[]) {
      this._tryChangeFilters(null, 'type', arrayToSet(v));
   }

   /**
    * Вынес повторяющийся код в отдельный метод
    */
   private _tryChangeFilters<K extends keyof IOperationLogFilters>(
      e: SyntheticEvent | null,
      field: K,
      value: IOperationLogFilters[K],
   ): void {
      const currentValue = this.props.value[field];

      if (!isEqual(value, currentValue)) {
         const newFilters = {
            ...this.props.value,
            [field]: value,
         };

         this.props.onChange(e, newFilters);
      }
   }
}
