import { autobind, classNames, CopyableText, formatDate, Json } from '@yandex-infracloud-ui/libs-next';
import * as H from 'history';
import { bool, func, instanceOf, object, string } from 'prop-types';
import * as React from 'react';
import { Children, SyntheticEvent } from 'react';
import { Link } from 'react-router-dom';
import { IOperationLogItem, ROUTE_LINKS } from '../../../models';
import { ClickOnlyUnmovedTrap, LocationContext } from '../../../shared';
import { OperationLogLayout } from '../models';
import styles from './OperationLogItem.module.css';

interface IProps {
   cls: string;
   expanded: Set<string>;
   hovered: boolean;
   layout: OperationLogLayout;
   value: IOperationLogItem;

   expand(id: string): void;

   onHover(auditLogId?: string): void;
}

interface IState {
   hovered: boolean;
}

export class OperationLogItem extends React.PureComponent<IProps, IState> {
   public static defaultProps = {
      cls: '',
      expanded: false,
   };

   public static propTypes = {
      cls: string,
      expand: func.isRequired,
      expanded: instanceOf(Set).isRequired,
      subItemsVisible: bool,
      toggleSubItems: func,
      value: object.isRequired,
   };

   constructor(props: IProps) {
      super(props);

      this.state = {
         hovered: false,
      };
   }

   public render() {
      const li = this.props.value;
      const expanded = this.props.expanded.has(li.id);

      const renderHostLink = this.props.layout !== 'host';
      const cells = (
         <>
            {this._renderTargetCell(li, renderHostLink)}
            {this._renderTypeCell(li)}
            {this._renderTimeCell(li)}
         </>
      );

      return (
         <>
            <ClickOnlyUnmovedTrap
               as={'tr'}
               onClick={this._expandRow}
               onMouseEnter={this._hoverOn}
               onMouseLeave={this._hoverOff}
               className={classNames(
                  {
                     [styles.subItem]: this.props.value.isSubItem,
                     [styles._expanded]: expanded,
                     [styles._hoveredSubItem]: this.state.hovered || this.props.hovered,
                  },
                  this.props.cls,
                  styles.logItem,
               )}
            >
               {cells}
            </ClickOnlyUnmovedTrap>

            {expanded && (
               <tr className={styles._additional} onMouseEnter={this._hoverOn} onMouseLeave={this._hoverOff}>
                  <td colSpan={Children.count(cells.props.children)}>
                     {this._renderLinks(li)}
                     {li.params ? <Json obj={li.params} /> : <pre>No params</pre>}
                  </td>
               </tr>
            )}
         </>
      );
   }

   @autobind
   private _expand() {
      this.props.expand(this.props.value.id);
   }

   @autobind
   private _expandRow(e: SyntheticEvent) {
      const target = e.target as EventTarget & Element;

      if (target.tagName !== 'TD' && target.tagName !== 'DIV') {
         return;
      }

      this._expand();
   }

   @autobind
   private _hoverOff() {
      const hovered = false;

      this.setState({ hovered });

      this.props.onHover(undefined);
   }

   @autobind
   private _hoverOn() {
      const hovered = true;

      this.setState({ hovered });

      this.props.onHover(this.props.value.audit_log_id);
   }

   private _renderLinkToScenario(li: IOperationLogItem, location: H.Location | null) {
      return li.scenario_id ? (
         <>
            <br />
            <Link
               to={{
                  pathname: ROUTE_LINKS.scenario(li.scenario_id),
                  state: { prev: location },
               }}
            >
               Scenario #{li.scenario_id}
            </Link>
         </>
      ) : null;
   }

   // noinspection JSMethodCanBeStatic
   private _renderLinks(li: IOperationLogItem) {
      const itemUri = `${ROUTE_LINKS.operationLogItem(li.id)}`;
      const auditLogUri =
         this.props.layout === 'host'
            ? ROUTE_LINKS.hostAuditLog(li.host_name, li.audit_log_id)
            : ROUTE_LINKS.auditLogItem(li.audit_log_id!);

      return (
         <div className={styles.links}>
            <span>
               <Link to={itemUri}>Link to this item</Link>

               <CopyableText text={`${window.location.origin}${itemUri}`} className={styles.copyIcon} />
            </span>

            <span>
               <Link to={auditLogUri}>Audit log</Link>

               <CopyableText text={`${window.location.origin}${auditLogUri}`} className={styles.copyIcon} />
            </span>
         </div>
      );
   }

   private _renderTargetCell(li: IOperationLogItem, renderHostLink: boolean) {
      const columnStyle = classNames(styles.targetColumn, styles.isGrouped);

      return (
         <td>
            <LocationContext.Consumer>
               {location => (
                  <div className={columnStyle}>
                     <div>
                        {renderHostLink ? (
                           <Link
                              to={{
                                 pathname: ROUTE_LINKS.hostOperationLog(li.host_name),
                                 state: { prev: location },
                              }}
                           >
                              {li.host_name}
                           </Link>
                        ) : (
                           li.host_name
                        )}

                        <div className={styles.secondString}>
                           Inv: {li.host_inv}
                           {this._renderLinkToScenario(li, location)}
                        </div>
                     </div>
                  </div>
               )}
            </LocationContext.Consumer>
         </td>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderTimeCell(li: IOperationLogItem) {
      return (
         <td>
            <div className={styles.dateColumn}>
               <span className={styles.date}>
                  <time className={styles.start}>{formatDate(li.time as Date)}</time>
               </span>
            </div>
         </td>
      );
   }

   // noinspection JSMethodCanBeStatic
   private _renderTypeCell(li: IOperationLogItem) {
      return (
         <td title={li.type} className={styles.typeColumn}>
            {li.type}
         </td>
      );
   }
}
