import { Suggest } from '@yandex-data-ui/common';
import {
   classNames,
   EMPTY_VALUE,
   HighlightedString,
   IListOption,
   useBehaviourSubject,
} from '@yandex-infracloud-ui/libs-next';
import Fuse from 'fuse.js';

import { globalState, OwnershipValue } from 'models';
import React, { SyntheticEvent, useCallback, useEffect, useMemo, useState } from 'react';
import { useSelector } from 'react-redux';

import { useProjectsLoadEnsure } from '../../shared';
import { RootState } from '../../state/store';
import commonClasses from '../form_fields/fields.module.css';

import classes from './ProjectSelect.module.css';

const projectsSelector = (s: RootState) => s.projects;

function useProjects(ownership: string): IListOption[] {
   const { allItems, isLoading } = useSelector(projectsSelector);

   const user = useBehaviourSubject(globalState.user); // TODO select from state.globals.user
   const [projects, setProjects] = useState<IListOption[]>([]);

   useProjectsLoadEnsure();

   useEffect(() => {
      if (user && !isLoading) {
         const projectIds = user.isAdmin || ownership === OwnershipValue.All ? allItems.allIds : user.projects;

         setProjects(
            projectIds
               .map(pId => allItems.byIds[pId])
               .filter(i => i !== undefined)
               .map(p => ({
                  name: p.name,
                  value: p.id,
               })),
         );
      }
   }, [allItems, isLoading, ownership, user]);

   return projects;
}

const FUSE_OPTIONS: Fuse.IFuseOptions<IListOption> = {
   keys: [
      {
         name: 'value',
         weight: 0.7,
      },
      {
         name: 'name',
         weight: 0.3,
      },
   ],
   shouldSort: true,
};

interface Props {
   disabled: boolean;
   error?: boolean;
   label?: string;
   name: string;
   onSelect: (e: SyntheticEvent | null, v: Set<string>) => void;
   ownership: OwnershipValue;
   placeholder?: string;
   readonly: boolean | undefined;
   value: any;
}

export const ProjectSelect = React.memo(
   ({ disabled, error, label, onSelect, ownership, placeholder, readonly, value }: Props) => {
      const allProjects = useProjects(ownership);
      const [query, setQuery] = useState(value ?? '');

      const fuse = useMemo(() => new Fuse(allProjects, FUSE_OPTIONS), [allProjects]);

      const getProjects = useCallback(() => (query ? fuse.search(query).map(r => r.item) : allProjects), [
         allProjects,
         fuse,
         query,
      ]);

      const getReadonlyValue = useCallback(() => {
         if (!value) {
            return EMPTY_VALUE;
         }

         const project = (allProjects || []).find(p => p.value === value);

         return project ? project.name : value;
      }, [value, allProjects]);

      const handleItemClick = useCallback(
         (v: IListOption) => {
            setQuery(v.value);
            onSelect(null, new Set([v.value]));
         },
         [onSelect],
      );

      return readonly ? (
         <div className={classes.readonly}>{getReadonlyValue()}</div>
      ) : (
         <Suggest
            disabled={disabled}
            getItems={getProjects}
            renderItem={item => (
               <div className={classes.item} key={item.name}>
                  <HighlightedString query={query} content={item.name} />
               </div>
            )}
            onItemClick={handleItemClick}
            placeholder={placeholder ?? label}
            debounce={250}
            className={classNames(classes.select, { [commonClasses.error]: error })}
            text={query}
            onUpdate={setQuery}
            showItemsOnNoText={true}
            getItemsOnMount={true}
         />
      );
   },
);

ProjectSelect.displayName = 'ProjectSelect';
