import { formatNumber, IListOption, IPureValueProps } from '@yandex-infracloud-ui/libs';
import { RadioButton } from '@yandex-data-ui/common';
import React, { useCallback } from 'react';

import { OwnershipValue } from '../../../models';
import { LegoButton } from '../../../shared/lego';

import classes from './OwnershipTumbler.module.css';
import { SaveIcon } from 'design/icons';

export interface ProjectCounts {
   [OwnershipValue.All]: number;
   [OwnershipValue.My]: number;
}

interface Props extends IPureValueProps<OwnershipValue> {
   defaultOwnership: OwnershipValue;
   projectCounts: ProjectCounts;

   onChangeDefault(v: OwnershipValue): void;
}

export function getOwnershipVariants(defaultOwnership: any | undefined) {
   return [
      { value: OwnershipValue.All, name: 'All' },
      { value: OwnershipValue.My, name: 'My' },
   ].sort(a => (a.value === defaultOwnership ? -1 : 1));
}

export const OwnershipTumbler: React.FC<Props> = React.memo(
   ({ defaultOwnership, onChangeDefault, projectCounts, value, onChange }) => {
      const ownershipVariants: IListOption[] = getOwnershipVariants(defaultOwnership);

      const changeOwnership = useCallback(
         (v: string) => {
            onChange(v as OwnershipValue);
         },
         [onChange],
      );

      const handleSave = useCallback(() => onChangeDefault(value), [onChangeDefault, value]);

      return (
         <div className={classes.tumbler}>
            <RadioButton onUpdate={changeOwnership} className={classes.radio} value={value}>
               {ownershipVariants.map(b => (
                  <RadioButton.Option key={b.value} value={b.value}>
                     {b.name}
                     <span className={classes.count}>{formatNumber(projectCounts[b.value])}</span>
                  </RadioButton.Option>
               ))}
            </RadioButton>

            <LegoButton
               cls={classes.saveBtn}
               theme={'clear'}
               onClick={handleSave}
               disabled={value === defaultOwnership}
               title={'Use by default'}
            >
               <SaveIcon />
            </LegoButton>
         </div>
      );
   },
);

OwnershipTumbler.displayName = 'OwnershipTumbler';
