import { classNames, IValueProps, toasts, useDebouncedValue } from '@yandex-infracloud-ui/libs';
import { Select } from 'lego-on-react';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import searchIcon from '../../../design/icons/search.svg';
import { OwnershipValue } from '../../../models';
import { config } from '../../../services';
import { LegoIcon, LegoTextInput } from '../../../shared/lego';
import {
   AutomationFilters,
   AutomationFilterValue,
   ProjectFilters,
   TypeFilters,
   TypeFilterValue,
} from '../../../state/projects/models';
import { OwnershipTumbler, ProjectCounts } from './OwnershipTumbler';

import classes from './ProjectFiltersView.module.css';

interface Props extends IValueProps<ProjectFilters> {
   className?: string;
   projectCounts: ProjectCounts;
}

export const ProjectFiltersView = React.memo(({ className = '', projectCounts, value, onChange }: Props) => {
   const [query, setQuery] = useState(value.query);
   const debouncedQuery = useDebouncedValue(query, 300);

   useEffect(() => {
      if (value.query !== debouncedQuery) {
         onChange(null, { ...value, query: debouncedQuery });
      }
   }, [debouncedQuery, onChange, value]);

   //region handlers
   const changeOwnership = useCallback(
      (v: OwnershipValue) => {
         onChange(null, { ...value, ownership: v });
      },
      [onChange, value],
   );

   const changeAutomation = useCallback(
      (val: string | string[]) => {
         const [automation] = val as AutomationFilterValue[];
         onChange(null, {
            ...value,
            automation,
         });
      },
      [onChange, value],
   );
   const changeType = useCallback(
      (val: string | string[]) => {
         const [type] = val as TypeFilterValue[];
         onChange(null, {
            ...value,
            type,
         });
      },
      [onChange, value],
   );

   const handleChangeDefaultOwnership = useCallback((v: OwnershipValue) => {
      config.ownershipDefault = v;
      toasts.info(
         v === OwnershipValue.My
            ? 'Now only hosts of your projects will be showed by default'
            : 'Now hosts of all projects will be showed by default',
         'Default project filter changed',
      );
   }, []);
   //endregion

   const ownershipFilter =
      projectCounts[OwnershipValue.My] > 0 ? (
         <label className={classes.field}>
            <span className={classes.fieldTitle}>Projects</span>

            <OwnershipTumbler
               value={value.ownership}
               onChange={changeOwnership}
               defaultOwnership={config.ownershipDefault}
               onChangeDefault={handleChangeDefaultOwnership}
               projectCounts={projectCounts}
            />
         </label>
      ) : null;

   return (
      <div className={classNames(classes.filters, className)}>
         <h2 className={classes.title}>Filters</h2>

         {ownershipFilter}

         <label className={classes.field}>
            <span className={classes.fieldTitle}>Automation</span>

            <Select
               theme={'normal'}
               size={'s'}
               type={'radio'}
               item-icon-hidden={true}
               val={value.automation}
               onChange={changeAutomation}
            >
               {AutomationFilters.map(f => (
                  <Select.Item key={f.value} val={f.value}>
                     {f.name}
                  </Select.Item>
               ))}
            </Select>
         </label>

         <label className={classes.field}>
            <span className={classes.fieldTitle}>Type</span>

            <Select
               theme={'normal'}
               size={'s'}
               type={'radio'}
               item-icon-hidden={true}
               val={value.type}
               onChange={changeType}
            >
               {TypeFilters.map(f => (
                  <Select.Item key={f.value} val={f.value}>
                     {f.name}
                  </Select.Item>
               ))}
            </Select>
         </label>

         <LegoTextInput
            cls={classes.query}
            iconLeft={<LegoIcon size={'s'} url={searchIcon} />}
            placeholder={'Enter project id, name or #tag'}
            text={query}
            onChange={setQuery}
         />
      </div>
   );
});

ProjectFiltersView.displayName = 'ProjectFiltersView';
