import { addHours } from 'date-fns';
import { ProjectItem } from '../../../state/projects/models';
import { getProjectWarning } from './ProjectItemView';

describe('rich_shared|ProjectSelect2/components/ProjectsItemView', () => {
   describe('getProjectWarning', () => {
      const item: ProjectItem = {
         dns_automation: { enabled: true },
         healing_automation: { enabled: true },
         id: 'test',
         name: 'Project name',
      };

      const fsmHandbrake = { timeout_time: addHours(new Date(), 1).getTime() / 1000 };
      const expiredFsmHandbrake = { timeout_time: addHours(new Date(), -1).getTime() / 1000 };

      it('should be empty if all is ok', () => {
         expect(getProjectWarning(item, 'short')).toBe('');

         expect(getProjectWarning(item, 'full')).toBe('');
      });

      it('should add warning for dns turned off', () => {
         const patchedItem = { ...item, dns_automation: { enabled: false } };

         expect(getProjectWarning(patchedItem, 'short')).toBe('DNS automation is off');

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'DNS automation is disabled, Wall-E will not fix DNS records automatically.',
         );
      });

      it('should add warning for healing turned off', () => {
         const patchedItem = { ...item, healing_automation: { enabled: false } };

         expect(getProjectWarning(patchedItem, 'short')).toBe('Healing automation is off');

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'Healing automation is disabled, Wall-E will not fix broken hosts automatically.',
         );
      });

      it('should add warning for both dns and healing turned off', () => {
         const patchedItem = {
            ...item,
            dns_automation: { enabled: false },
            healing_automation: { enabled: false },
         };

         expect(getProjectWarning(patchedItem, 'short')).toBe('DNS and healing automations are off');

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'Healing and DNS automations are disabled,' +
               ' Wall-E will not fix broken hosts and DNS records automatically.',
         );
      });

      it('should add warning if FSM handbrake is on', () => {
         const patchedItem = {
            ...item,
            fsm_handbrake: fsmHandbrake,
         };

         expect(getProjectWarning(patchedItem, 'short')).toBe('FSM-handbrake is on');

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'FSM-handbrake is enabled, Wall-E will stop processing any host tasks.',
         );
      });

      it('should add warning if FSM handbrake is on and dns turned off', () => {
         const patchedItem = {
            ...item,
            dns_automation: { enabled: false },
            fsm_handbrake: fsmHandbrake,
         };

         expect(getProjectWarning(patchedItem, 'short')).toBe('DNS automation is off, FSM-handbrake is on');

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'DNS automation is disabled and FSM-handbrake is enabled, ' +
               'Wall-E will not fix DNS records and stop processing any host tasks.',
         );
      });

      it('should add warning if FSM handbrake is on and dns and healing turned off', () => {
         const patchedItem = {
            ...item,
            dns_automation: { enabled: false },
            fsm_handbrake: fsmHandbrake,
            healing_automation: { enabled: false },
         };
         expect(getProjectWarning(patchedItem, 'short')).toBe(
            'DNS and healing automations are off, FSM-handbrake is on',
         );

         expect(getProjectWarning(patchedItem, 'full')).toBe(
            'Healing and DNS automations are disabled and FSM-handbrake is enabled, ' +
               'Wall-E will not fix broken hosts and DNS records automatically and stop processing any host tasks.',
         );
      });

      it('should not add warning if FSM handbrake is on but expired', () => {
         const patchedItem = { ...item, fsm_handbrake: expiredFsmHandbrake };

         expect(getProjectWarning(patchedItem, 'short')).toBe('');

         expect(getProjectWarning(patchedItem, 'full')).toBe('');
      });
   });
});
