import { classNames } from '@yandex-infracloud-ui/libs';
import { CoqIcon } from 'design/icons';
import * as React from 'react';
import { CSSProperties, SyntheticEvent, useCallback } from 'react';
import { Link } from 'react-router-dom';

import { ROUTE_LINKS } from '../../../models';
import { LegoCheckbox } from '../../../shared';
import { ProjectItem } from '../../../state/projects/models';
import classes from './ProjectItemView.module.css';

interface IProps {
   item: ProjectItem;
   selected: boolean;
   style?: CSSProperties;

   toggle(itemId: string): void;
}

interface ProjectWarningRecord {
   full: string;
   short: string;
}

/**
 * Сообщения для всех возможных комбинаций флагов автоматики
 *
 * Ключом является число с битовыми флагами. Порядок бит (слева): DNS, Healing, FSM
 */
const warnings = new Map<number, ProjectWarningRecord>([
   // 000 all is ok
   [0, { short: '', full: '' }],

   // 001 only fsm
   [
      1,
      {
         full: 'FSM-handbrake is enabled, Wall-E will stop processing any host tasks.',
         short: 'FSM-handbrake is on',
      },
   ],

   // 010 only healing
   [
      2,
      {
         full: 'Healing automation is disabled, Wall-E will not fix broken hosts automatically.',
         short: 'Healing automation is off',
      },
   ],

   // 011 healing + fsm
   [
      3,
      {
         full:
            'Healing automation is disabled and FSM-handbrake is enabled, ' +
            'Wall-E will not fix broken hosts and stop processing any host tasks.',
         short: 'Healing automation is off, FSM-handbrake is on',
      },
   ],

   // 100 only dns
   [
      4,
      {
         full: 'DNS automation is disabled, Wall-E will not fix DNS records automatically.',
         short: 'DNS automation is off',
      },
   ],

   // 101 dns + fsm
   [
      5,
      {
         full:
            'DNS automation is disabled and FSM-handbrake is enabled, ' +
            'Wall-E will not fix DNS records and stop processing any host tasks.',
         short: 'DNS automation is off, FSM-handbrake is on',
      },
   ],

   // 110 dns + healing
   [
      6,
      {
         full:
            'Healing and DNS automations are disabled, ' +
            'Wall-E will not fix broken hosts and DNS records automatically.',
         short: 'DNS and healing automations are off',
      },
   ],

   // 111 dns + healing + fsm
   [
      7,
      {
         full:
            'Healing and DNS automations are disabled and FSM-handbrake is enabled, ' +
            'Wall-E will not fix broken hosts and DNS records automatically and stop processing any host tasks.',
         short: 'DNS and healing automations are off, FSM-handbrake is on',
      },
   ],
]);

/**
 * Возвращает строчку с предупреждениями об отключенной автоматике или включенном FSM Handbrake
 *
 * Функция грязная, т.к. вызывает Date.now()
 */
export function getProjectWarning(item: ProjectItem, form: 'short' | 'full'): string {
   const d = item.dns_automation && !item.dns_automation.enabled;
   const h = item.healing_automation && !item.healing_automation.enabled;
   const f = item.fsm_handbrake && item.fsm_handbrake.timeout_time * 1000 > Date.now();

   // binary dhf
   const b = (d ? 4 : 0) + (h ? 2 : 0) + (f ? 1 : 0);

   return warnings.get(b)![form];
}

export const ProjectItemView = React.memo(({ item, selected, style, toggle }: IProps) => {
   const toggleItem = useCallback(
      (e: SyntheticEvent | null) => {
         if (e) {
            e.preventDefault();
            e.stopPropagation();
         }

         toggle(item.id);
      },
      [item, toggle],
   );

   const stopEvent = useCallback((e: React.MouseEvent) => e.stopPropagation(), []);

   const fakeToggle = useCallback(() => null, []);

   const warning = getProjectWarning(item, 'short');

   let title = `id: ${item.id}`;
   if (item.tags) {
      title += `, tags: ${item.tags.map(t => `#${t}`).join(' ')}`;
   }

   return (
      <div
         style={style}
         className={classNames(classes.item, {
            [classes.selected]: selected,
         })}
         onClick={toggleItem}
         title={title}
      >
         <LegoCheckbox cls={classes.name} checked={selected} onChange={fakeToggle} text={item.name} />

         <div className={classes.link}>
            <Link to={ROUTE_LINKS.project(item.id)} title={'Project details'} onClick={stopEvent}>
               <CoqIcon />
            </Link>
         </div>

         {warning ? <div className={classes.warnings}>{warning}</div> : null}
      </div>
   );
});

ProjectItemView.displayName = 'ProjectItemView';
