import { classNames } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useRef } from 'react';
import AutoSizer from 'react-virtualized-auto-sizer';

import { areEqual, VariableSizeList } from 'react-window';

import { LegoCheckbox } from '../../../shared/lego';
import { IdSet } from '../../../state/commonModels';
import { ProjectItem } from '../../../state/projects/models';

import { ProjectItemView } from './ProjectItemView';
import classes from './ProjectList.module.css';

interface Props {
   className?: string;
   isAllSelected: boolean;
   items: ProjectItem[];
   selected: IdSet;

   toggle(itemId: string): void;

   toggleAll(): void;
}

const Row = React.memo(({ index, style, data: { items, selected, toggle } }: any) => {
   const p = items[index];

   return <ProjectItemView style={style} item={p} selected={selected[p.id]} toggle={toggle} />;
}, areEqual);

export const ProjectList = React.memo(
   ({ className = '', items, selected, isAllSelected, toggle, toggleAll }: Props) => {
      //region hooks
      const listRef = useRef<VariableSizeList | null>(null);
      const data = { items, selected, toggle };

      type Data = typeof data;
      //endregion

      //region effects
      useEffect(() => {
         if (listRef.current) {
            if (items.length > 0) {
               listRef.current.scrollToItem(0);
            }
            listRef.current.resetAfterIndex(0, true);
         }
      }, [items]);
      //endregion

      //region handlers
      const getItemKey = useCallback((index: number, d: Data) => d.items[index].id, []);

      const getItemHeight = useCallback(
         (index: number) => {
            const item = items[index];

            const d = item.dns_automation ? !item.dns_automation.enabled : false;
            const h = item.healing_automation ? !item.healing_automation.enabled : false;
            const a = d || h;
            const f = Boolean(item.fsm_handbrake);

            if (a && f) {
               // Максимальная высота, обе надписи с ошибкой
               return 84;
            }

            if (a || f) {
               // Только одна ошибка
               return 64;
            }

            // Без ошибок;
            return 46;
         },
         [items],
      );
      //endregion

      //region render
      return (
         <div className={classNames(classes.list, className)}>
            <div className={classes.selectAllWrapper}>
               <LegoCheckbox
                  id={'selectAllProjects'}
                  text={'Select all'}
                  checked={isAllSelected}
                  onChange={toggleAll}
                  cls={classNames(classes.selectAll, { [classes.selectedAll]: isAllSelected })}
               />
            </div>

            <div className={classes.listWrapper}>
               <AutoSizer>
                  {({ height, width }) => (
                     <VariableSizeList
                        ref={listRef}
                        estimatedItemSize={55}
                        itemData={data}
                        width={width}
                        height={height}
                        itemCount={items.length}
                        overscanCount={10}
                        itemSize={getItemHeight}
                        itemKey={getItemKey}
                     >
                        {Row}
                     </VariableSizeList>
                  )}
               </AutoSizer>
            </div>
         </div>
      );
      //endregion
   },
);

ProjectList.displayName = 'ProjectList';
