import { classNames, IListOption } from '@yandex-infracloud-ui/libs';
import { Button, Icon } from 'lego-on-react';
import React, { SyntheticEvent } from 'react';
import { Link } from 'react-router-dom';
import { ROUTE_LINKS } from '../../../models';

import classes from './ProjectListWithTagLayout.module.css';

interface Props {
   className?: string;
   editable: boolean;
   items: IListOption[];

   onRemove?(id: string): void;
}

/**
 * Отображает список проектов в виде списка тегов, только с ссылками на проекты.
 * Практически копипаста с TagList2.
 */
export const ProjectListWithTagLayout: React.FC<Props> = React.memo(
   ({ className = '', editable, items, onRemove }: Props) => {
      if (editable && !onRemove) {
         throw new Error('Expected onRemove prop if TagList2 is editable');
      }

      const handleRemove = (e: SyntheticEvent, item: IListOption) => {
         e.preventDefault();
         e.stopPropagation();
         onRemove!(item.value);
      };

      return (
         <div className={classNames(classes.tagList, className)}>
            {items.map(item => (
               <Link
                  key={item.value}
                  to={ROUTE_LINKS.project(item.value)}
                  className={classNames(classes.tag, { [classes.editable]: editable })}
                  title={`${item.name}${item.note ? `: ${item.note}` : ''}`}
               >
                  {item.name}

                  {editable ? (
                     <Button
                        theme={'clear'}
                        size={'xs'}
                        icon={<Icon type={'close'} size={'xs'} />}
                        cls={classes.removeButton}
                        onClick={e => handleRemove(e, item)}
                     />
                  ) : null}
               </Link>
            ))}
         </div>
      );
   },
);

ProjectListWithTagLayout.displayName = 'ProjectListWithTagLayout';
