import { fromQuery } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useMemo, useRef } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { useHistory, useLocation } from 'react-router-dom';

import { OwnershipValue } from '../../../models';
import { config } from '../../../services';
import { useProjectsLoadEnsure } from '../../../shared';
import { projectsSlice } from '../../../state/projects';
import { ProjectFilters } from '../../../state/projects/models';
import { RootState } from '../../../state/store';
import { patchUrlQuery } from '../../../utils/toLibs';

import { ProjectFiltersView } from './ProjectFiltersView';
import { ProjectList } from './ProjectList';
import classes from './ProjectSelect2.module.css';
import { ProjectSelect2Skeleton } from './ProjectSelect2Skeleton';

interface ProjectSelectUrlParams {
   ownership?: OwnershipValue.All;
}

export const ProjectSelect2 = React.memo(() => {
   //region hooks
   const userProjects = useSelector((s: RootState) => s.globals.user?.projects || []); // TODO

   const {
      allItems,
      filteredIds,
      filters,
      isAllSelected,
      isLoading,
      selectedIds,
      // userProjects,
   } = useSelector((s: RootState) => s.projects);
   const dispatch = useDispatch();
   //endregion

   //region effects
   useProjectsLoadEnsure();

   useOwnershipUrlUpdater(filters.ownership);
   //endregion

   //region handlers
   const updateFilters = useCallback(
      (e: SyntheticEvent | null, f: ProjectFilters) => {
         dispatch(projectsSlice.actions.setFilters(f));
      },
      [dispatch],
   );

   const toggle = useCallback(
      (itemId: string) => {
         dispatch(projectsSlice.actions.toggleItem(itemId));
      },
      [dispatch],
   );

   const toggleAll = useCallback(() => {
      dispatch(projectsSlice.actions.toggleAllItems());
   }, [dispatch]);
   //endregion

   //region render
   const projectCounts = {
      [OwnershipValue.All]: allItems.allIds.length,
      [OwnershipValue.My]: userProjects.length,
   };

   const projects = useMemo(() => filteredIds.map(id => allItems.byIds[id]!), [filteredIds, allItems]);

   return (
      <div className={classes.wrapper}>
         {isLoading ? (
            <ProjectSelect2Skeleton />
         ) : (
            <>
               <ProjectFiltersView
                  className={classes.filters}
                  projectCounts={projectCounts}
                  value={filters}
                  onChange={updateFilters}
               />

               <ProjectList
                  className={classes.list}
                  items={projects}
                  selected={selectedIds}
                  isAllSelected={isAllSelected}
                  toggle={toggle}
                  toggleAll={toggleAll}
               />
            </>
         )}
      </div>
   );
   //endregion
});

ProjectSelect2.displayName = 'ProjectSelect2';

function useOwnershipUrlUpdater(ownershipFromState: OwnershipValue) {
   const history = useHistory();
   const location = useLocation<ProjectSelectUrlParams>();
   const dispatch = useDispatch();

   const skipNextUrlUpdate = useRef(false); // Чтобы развязать циклическое обновление

   // update filters after url changing
   useEffect(() => {
      const urlParams: ProjectSelectUrlParams = fromQuery(location.search || '');
      const ownership = urlParams.ownership || config.ownershipDefault;

      skipNextUrlUpdate.current = true;
      dispatch(projectsSlice.actions.setOwnershipFilter(ownership));
   }, [dispatch, location]);

   // update url after filters changing
   useEffect(() => {
      if (skipNextUrlUpdate.current) {
         skipNextUrlUpdate.current = false;

         return;
      }

      patchUrlQuery(history, location, {
         ownership: ownershipFromState === config.ownershipDefault ? '' : ownershipFromState,
      });
   }, [ownershipFromState, history, location]);
}
