import { classNames, formatNumber, isEqual, plural } from '@yandex-infracloud-ui/libs';
import { Icon } from 'lego-on-react';
import * as React from 'react';
import { useMemo } from 'react';
import warnSvg from '../../../design/icons/exclamation-triangle.svg';
import { ProjectItem } from '../../../state/projects/models';
import { getProjectWarning } from './ProjectItemView';

import classes from './ProjectWarnings2.module.css';

interface Props {
   className?: string;
   projects: ProjectItem[];

   onClick?(): void;
}

interface AutomationWarningProps {
   action: string;
   count: number;
   subject: string;
}

const AutomationWarning = React.memo(({ count, subject, action }: AutomationWarningProps) =>
   count > 0 ? (
      <span className={classes.warningRow}>
         {subject} {plural(count, 'is', 'are')} {action}&nbsp;
         <span className={classes.count}>
            {formatNumber(count)}&nbsp;
            {plural(count, 'project', 'projects')}
         </span>
      </span>
   ) : null,
);

AutomationWarning.displayName = 'AutomationWarning';

export const ProjectWarnings2: React.FC<Props> = React.memo(({ className = '', children, projects, onClick }) => {
   const counters = useMemo(() => {
      const now = Date.now() / 1000; // HACK

      return projects.reduce(
         (acc, item) => {
            if (item.fsm_handbrake && item.fsm_handbrake.timeout_time > now) {
               acc.fsm += 1;
            }
            if (item.dns_automation && !item.dns_automation.enabled) {
               acc.dns += 1;
            }
            if (item.healing_automation && !item.healing_automation.enabled) {
               acc.healing += 1;
            }

            return acc;
         },
         { fsm: 0, dns: 0, healing: 0 },
      );
   }, [projects]);

   if (isEqual(counters, { fsm: 0, dns: 0, healing: 0 })) {
      return <>{children}</>;
   }

   if (projects.length === 0) {
      return null; // TODO выводить глобальные автоматики
   }

   if (projects.length === 1) {
      return (
         <section
            className={classNames(classes.warning, className, {
               [classes.warningWithTumblers]: Boolean(children),
            })}
         >
            <div className={classes.icon}>
               <Icon url={warnSvg} style={{ width: 16 }} />
            </div>
            <div className={classes.text}>
               <h3 className={classes.title}>{getProjectWarning(projects[0]!, 'short')}</h3>
               <p className={classes.notes}>{getProjectWarning(projects[0]!, 'full')}</p>
            </div>
            {children ? (
               <section className={classes.tumblers}>{children}</section>
            ) : (
               <div className={classes.buttons}>
                  <button className={classes.button} onClick={onClick}>
                     Show tumblers
                  </button>
               </div>
            )}
         </section>
      );
   }

   return (
      <section className={classNames(classes.warning, className)}>
         <div className={classes.icon}>
            <Icon url={warnSvg} style={{ width: 16 }} />
         </div>
         <div className={classes.text}>
            <h3 className={classes.title}>Automations are turned off</h3>
            <p className={classes.notes}>
               <AutomationWarning subject={'Healing automation'} count={counters.healing} action={'off'} />
               <AutomationWarning subject={'DNS-automation'} count={counters.dns} action={'off'} />
               <AutomationWarning subject={'FSM-handbrake'} count={counters.fsm} action={'applied'} />
            </p>
         </div>
         <div className={classes.buttons}>
            <button className={classes.button} onClick={onClick}>
               Show tumblers
            </button>
         </div>
      </section>
   );
});

ProjectWarnings2.displayName = 'ProjectWarnings2';
