import { DISMISS_REASON, IListOption, modalService, toasts, useDismounted } from '@yandex-infracloud-ui/libs';
import { CoqIcon } from 'design/icons';
import * as React from 'react';
import { useCallback, useMemo, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { Link } from 'react-router-dom';
import { takeUntil } from 'rxjs/operators';
import { ROUTE_LINKS } from '../../../models';
import { idSetToList } from '../../../state/commonModels';
import { loadAutomations } from '../../../state/fullProjectForm';
import { projectsSlice } from '../../../state/projects';
import { RootState } from '../../../state/store';
import { ProjectListAutomationTumblersModal } from '../../project_automations';
import { ProjectListWithTagLayout } from './ProjectListWithTagLayout';

import classes from './ProjectsHeader.module.css';
import { ProjectWarnings2 } from './ProjectWarnings2';

interface Props extends RouteComponentProps {
   className?: string;
}

export const ProjectsHeader = React.memo(({ className = '' }: Props) => {
   //region hooks
   const dismounted = useDismounted();
   const [expanded, setExpanded] = useState(false);
   const { selectedIds, allItems, title } = useSelector((s: RootState) => s.projects);
   const dispatch = useDispatch();

   const projects = useMemo(() => {
      const ids = idSetToList(selectedIds);
      ids.sort();

      return ids.map(id => allItems.byIds[id] || { id, name: id });
   }, [allItems.byIds, selectedIds]);
   //endregion

   const isEmpty = projects.length === 0;
   const isSingle = projects.length === 1;

   //region handlers
   const toggle = useCallback(() => setExpanded(e => !e), []);

   const deselectProject = useCallback(
      (id: string) => {
         dispatch(projectsSlice.actions.toggleItem(id));
      },
      [dispatch],
   );

   const showTumblers = useCallback(() => {
      modalService
         .open(ProjectListAutomationTumblersModal, { items: projects }, true)
         .pipe(takeUntil(dismounted))
         .subscribe(
            projectId => dispatch(loadAutomations(projectId, dismounted)),
            reason => {
               if (reason !== DISMISS_REASON) {
                  toasts.apiError('Project automation toggling', reason);
               }
            },
         );
   }, [dismounted, dispatch, projects]);
   //endregion

   //region render
   const tagItems = projects.map(project => ({ name: project.name, value: project.id } as IListOption));

   return (
      <div className={className}>
         {isEmpty ? (
            <div className={classes.title}>
               <h1>{title}</h1>
            </div>
         ) : isSingle ? (
            <div className={classes.title}>
               <h1>{title}</h1>

               <Link
                  to={ROUTE_LINKS.project(projects[0].id)}
                  className={classes.projectLinkButton}
                  title={'Project details'}
               >
                  <CoqIcon />
               </Link>
            </div>
         ) : (
            <>
               <div className={classes.title}>
                  <h1>{title}</h1>
                  <button className={classes.hideAllButton} onClick={toggle}>
                     <span>{expanded ? 'Hide all' : 'Show all'}</span>
                  </button>
               </div>

               {expanded ? (
                  <ProjectListWithTagLayout
                     className={classes.projects}
                     items={tagItems}
                     editable={true}
                     onRemove={deselectProject}
                  />
               ) : null}
            </>
         )}

         <ProjectWarnings2 className={classes.warnings} projects={projects} onClick={showTumblers} />
      </div>
   );
   //endregion
});

ProjectsHeader.displayName = 'ProjectsHeader';
