import { EMPTY_VALUE, FieldLayout, IFieldProps, toasts, useDismounted, isEmpty } from '@yandex-infracloud-ui/libs';
import { Select } from 'lego-on-react';
import { default as React, useEffect, useState } from 'react';
import { map, takeUntil } from 'rxjs/operators';

import { IConstants, NONE_ITEM } from '../../models';
import { dictApi } from '../../services';

export interface IConstantSelectConfig {
   defaultPlaceholder: string;
   extractor: (c: IConstants) => string[] | number[];
   withNone?: boolean;
}

export const ConstantSelectField = React.memo(
   ({ config, help, error, disabled, readonly, value, onChange }: IFieldProps<string>) => {
      const subConfig = config.subConfig as IConstantSelectConfig;
      if (!subConfig) {
         throw new Error('Expected defined subConfig for ConstantSelectField');
      }

      // hooks
      const dismounted = useDismounted();
      const [options, setOptions] = useState<string[]>([]);

      // effects
      useEffect(() => {
         dictApi
            .getConstants()
            .pipe(map(subConfig.extractor), takeUntil(dismounted))
            .subscribe((items: any) => {
               if (subConfig.withNone) {
                  items = [NONE_ITEM, ...items];
               }
               setOptions(items as string[]);
            }, toasts.handleApiError('Constants loading'));
      }, [dismounted, subConfig.extractor, subConfig.withNone]);

      // handlers
      const updateValue = (v: string | string[]) => {
         onChange(null, v[0]);
      };

      // render
      return (
         <FieldLayout config={config} help={help} error={error}>
            {disabled || readonly ? (
               <>{!isEmpty(value) ? value : EMPTY_VALUE}</>
            ) : (
               // TODO заменить Select на Dropdown из ui-components
               <div data-e2e={'ConstantSelectField:' + config.name}>
                  <Select
                     theme={'normal'}
                     size={'s'}
                     type={'radio'}
                     placeholder={config.placeholder || subConfig.defaultPlaceholder}
                     val={value}
                     onChange={updateValue}
                  >
                     {options.map(o => (
                        <Select.Item key={o} val={o}>
                           {o}
                        </Select.Item>
                     ))}
                  </Select>
               </div>
            )}
         </FieldLayout>
      );
   },
);

ConstantSelectField.displayName = 'ConstantSelectField';

export const profileConfigSubConfig: IConstantSelectConfig = {
   defaultPlaceholder: 'Select profile config',
   extractor: c => c.allowed_default_profiles,
};

export const ipMethodSubConfig: IConstantSelectConfig = {
   defaultPlaceholder: 'Select Host-id for IP-address',
   extractor: c => c.mtn_ip_methods,
};
