import {
   FieldLayout,
   formatNumber,
   getSetDifference,
   IFieldProps,
   Loader,
   mergeSets,
   plural,
   toggleSetItem,
   useBehaviourSubject,
} from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { ReactNode, SyntheticEvent, useCallback, useEffect, useMemo, useState } from 'react';

import { globalState, isProjectOwnersEmpty, ProjectOwners } from '../../models';
import { auth, config } from '../../services';
import { OwnersList, UserSelect, UserSelectMode } from '../../shared';

import classes from './ProjectOwnersField.module.css';

// Вынес логику переключения состояния в кастомный хук ради того, чтобы не раздувать компонент
function useProjectOwnersEditor(actualValue: ProjectOwners | undefined) {
   const [value, setValue] = useState(actualValue);

   useEffect(() => setValue(actualValue), [actualValue]);

   const onAdd = useCallback(
      (e: SyntheticEvent | null, login: string) => {
         if (!value) {
            return;
         }

         // уже добавлен
         if (
            login === null ||
            value.actualOwners.has(login) ||
            value.requestedOwners.has(login) ||
            value.revokingOwners.has(login) ||
            value.toAdd.has(login)
         ) {
            return;
         }

         // Частный случай: пытаемся добавить пользователя, которого уже удалили
         if (login && value.toRemove.has(login)) {
            setValue({
               ...value,
               actualOwners: toggleSetItem(value.actualOwners, login),
               toRemove: toggleSetItem(value.toRemove, login),
            });

            return;
         }

         setValue({
            ...value,
            toAdd: toggleSetItem(value.toAdd, login),
         });
      },
      [value],
   );

   const onActualChange = useCallback(
      (e: SyntheticEvent | null, v: Set<string>) => {
         if (!value) {
            return;
         }

         const { removed } = getSetDifference(value.actualOwners, v);
         const toRemove = mergeSets(value.toRemove, removed);

         setValue({
            ...value,
            actualOwners: v,
            toRemove,
         });
      },
      [value],
   );

   const onToAddChange = useCallback(
      (e: SyntheticEvent | null, v: Set<string>) => {
         if (!value) {
            return;
         }

         setValue({
            ...value,
            toAdd: v,
         });
      },
      [value],
   );

   const onToRemoveChange = useCallback(
      (e: SyntheticEvent | null, v: Set<string>) => {
         if (!value) {
            return;
         }

         const { removed } = getSetDifference(value.toRemove, v);
         const actualOwners = mergeSets(value.actualOwners, removed);

         setValue({
            ...value,
            actualOwners,
            toRemove: v,
         });
      },
      [value],
   );

   return {
      onActualChange,
      onAdd,
      onToAddChange,
      onToRemoveChange,
      value,
   };
}

/**
 * Редактируемое поле владельцев проекта.
 *
 * Содержит сложную логику, т.к. критериев, влияющих на состояние много:
 *    Может ли пользователь редактировать проект (да/нет)
 *    Включен ли режим редактирования (да/нет)
 *    Различные комбинации в value:
 *       Если всё пусто (маловероятно, но все ж) - вывести "No owners"
 *       Варианты пустых списков владельцев для каждого из Set.
 *
 * toAdd и toRemove показываются только в режиме редактирования (это отрабатывать не нужно, положимся на внешнюю логику)
 * Из списка toAdd и toRemove можно удалять.
 * Значение из toRemove в этом случае должно возвращаться туда, откуда было удалено.
 *
 * Порядок:
 *    1. Поле для ввода нового овнера (в режиме редактирования)
 *    2. Список действующих овнеров
 *    3. Список овнеров, ожидающих подтверждение в IDM
 *    4. Список отозванных овенров, но еще не удаленных в IDM окончательно
 *    5. Список овнеров, которые будут добавлены при сабмите формы
 *    6. Список овнеров, которые будут удалены при сабмите формы
 *    7. Если все списки пусты и режим просмотра - выводим No owners
 *    8. Если мы не можем редактировать проект, то можем запросить доступ через форму IDM (оставить ссылку).
 */
export const ProjectOwnersField: React.FC<IFieldProps<ProjectOwners | undefined>> = React.memo(
   ({
      config: fieldConfig,
      context: { projectId, isOwnersSaving: isSaving },
      disabled,
      error,
      help,
      readonly,
      value: externalValue,
      onChange,
   }) => {
      //region hooks
      const user = useBehaviourSubject(globalState.user);
      const { value, onAdd, onActualChange, onToAddChange, onToRemoveChange } = useProjectOwnersEditor(externalValue);
      //endregion

      //region effects
      // Обновление внешнего значения при обновлении внутреннего
      useEffect(() => onChange(null, value), [onChange, value]);
      //endregion

      //region render
      const label = useMemo((): ReactNode => {
         if (typeof fieldConfig.label === 'string') {
            return isSaving ? <Loader text={`${fieldConfig.label} saving`} align={'left'} /> : fieldConfig.label;
         }

         return (
            <>
               {fieldConfig.label}
               {isSaving ? <Loader text={'Saving'} align={'left'} /> : null}
            </>
         );
      }, [fieldConfig.label, isSaving]);

      if (value === undefined) {
         return (
            <FieldLayout config={fieldConfig} help={help} error={error} customLabel={label}>
               <Loader text={'Project owners are loading'} />
            </FieldLayout>
         );
      }

      const editable = !disabled && !readonly && !isSaving;

      const isInOwners =
         auth.canEditProject({ id: projectId }, user) || (user ? value.requestedOwners.has(user.login) : false);

      return (
         <FieldLayout config={fieldConfig} help={help} error={error} customLabel={label}>
            {
               // 1. Поле нового овнера
               editable ? (
                  // TODO Отключил автофокус, пока не починят Лего
                  //  https://lego-team.slack.com/messages/C0DL13NJY/convo/C0DL13NJY-1543422059.494500/

                  <div className={classes.row}>
                     <UserSelect
                        mode={UserSelectMode.Owners}
                        name={fieldConfig.name}
                        onSelect={onAdd}
                        autofocus={false}
                     />
                  </div>
               ) : null
            }

            {
               // 2. Список действующих овнеров
               value.actualOwners.size > 0 ? (
                  <div className={classes.row}>
                     <OwnersList editable={editable} value={value.actualOwners} onChange={onActualChange} />
                  </div>
               ) : null
            }

            {
               // 3. Список овнеров, ожидающих подтверждение в IDM
               value.requestedOwners.size > 0 ? (
                  <div className={classes.row}>
                     <label>
                        Access was requested via IDM for&nbsp;
                        {formatNumber(value.requestedOwners.size)}&nbsp;
                        {plural(value.requestedOwners.size, 'person', 'people')}:
                     </label>

                     <OwnersList editable={false} value={value.requestedOwners} />
                  </div>
               ) : null
            }

            {
               // 4. Список отозванных овенров, но еще не удаленных в IDM окончательно
               value.revokingOwners.size > 0 ? (
                  <div className={classes.row}>
                     <label>
                        Access is revoking via IDM for&nbsp;
                        {formatNumber(value.revokingOwners.size)}&nbsp;
                        {plural(value.revokingOwners.size, 'person', 'people')}:
                     </label>

                     <OwnersList editable={false} value={value.revokingOwners} />
                  </div>
               ) : null
            }

            {
               // 5. Список овнеров, которые будут добавлены при сабмите формы (form submission)
               value.toAdd.size > 0 ? (
                  <div className={classes.row}>
                     <label>
                        Access will be requested via IDM after the project saving for&nbsp;
                        {formatNumber(value.toAdd.size)}&nbsp;
                        {plural(value.toAdd.size, 'person', 'people')}:
                     </label>

                     <OwnersList editable={editable} forceAvatars={true} value={value.toAdd} onChange={onToAddChange} />
                  </div>
               ) : null
            }

            {
               // 6. Список овнеров, которые будут удалены при сабмите формы
               value.toRemove.size > 0 ? (
                  <div className={classes.row}>
                     <label>
                        Access revocation will be requested after the project saving for&nbsp;
                        {formatNumber(value.toRemove.size)}&nbsp;
                        {plural(value.toRemove.size, 'person', 'people')}:
                     </label>

                     <OwnersList
                        editable={editable}
                        forceAvatars={true}
                        value={value.toRemove}
                        onChange={onToRemoveChange}
                     />
                  </div>
               ) : null
            }

            {
               // 7. Если все списки пусты и режим просмотра - выводим No owners
               !editable && isProjectOwnersEmpty(value) ? <div className={classes.row}>No owners</div> : null
            }

            {
               // 8. Если мы не можем редактировать проект, то можем запросить доступ через форму IDM (оставить ссылку)
               !isInOwners && user ? (
                  <div className={classes.row}>
                     <a
                        href={config.getIdmAddOwnerUrl(projectId, user.login)}
                        target={'_blank'}
                        rel={'noopener noreferrer'}
                     >
                        Request access for me
                     </a>
                  </div>
               ) : null
            }
         </FieldLayout>
      );
      //endregion
   },
);

ProjectOwnersField.displayName = 'ProjectOwnersField';
