import {
   fixedMemo,
   getSetDifference,
   IUserOrGroup,
   plural,
   toasts,
   useDismounted,
   UserList,
   UserOrGroupType,
} from '@yandex-infracloud-ui/libs';
import React, { SyntheticEvent, useCallback, useMemo } from 'react';
import { forkJoin } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import { projectApi } from '../../../services';
import { UserSelect, UserSelectMode } from '../../../shared';

import classes from './RoleEditor.module.css';

interface Props {
   disabled: boolean;
   projectId: string;
   readonly: boolean;
   role: string;
   value: string[];

   onChange(role: string, value: string[]): void;
}

export const RoleEditor: React.FC<Props> = fixedMemo(({ value, onChange, role, projectId, disabled, readonly }) => {
   const editable = !disabled && !readonly;
   const dismounted = useDismounted();

   const internalValue = useMemo(
      () =>
         value.map(
            v =>
               ({
                  id: v.startsWith('@') ? v.replace(/^@/, '') : v,
                  name: v,
                  type: v.startsWith('@') ? UserOrGroupType.Group : UserOrGroupType.People,
               } as IUserOrGroup),
         ),
      [value],
   );

   const handleSelect = useCallback(
      (e: SyntheticEvent, login: string | null) => {
         if (!(login && !value.includes(login))) {
            return;
         }

         projectApi
            .addRoleMember(projectId, role, login)
            .pipe(takeUntil(dismounted))
            .subscribe(members => {
               onChange(role, [...value, login]);
               toasts.success(`${members} has been added to role "${role}`);
            }, toasts.handleApiError(`Adding member for project role "${role}"`));
      },
      [dismounted, onChange, projectId, role, value],
   );

   const handleChangeList = useCallback(
      (e: SyntheticEvent, v: IUserOrGroup[]) => {
         const { removed } = getSetDifference(new Set(value), new Set(v.map(i => i.id)));

         forkJoin(Array.from(removed).map(member => projectApi.removeRoleMember(projectId, role, member)))
            .pipe(takeUntil(dismounted))
            .subscribe(members => {
               const newValue = value.filter(i => !removed.has(i));

               onChange(role, newValue);
               toasts.success(
                  `${members.join(', ')} ${plural(members.length, 'has', 'have')} been removed from role "${role}`,
               );
            }, toasts.handleApiError(`Removing member for project role "${role}"`));
      },
      [dismounted, onChange, projectId, role, value],
   );

   return (
      <div className={classes.wrapper}>
         {editable ? (
            <UserSelect cls={classes.input} mode={UserSelectMode.Owners} onSelect={handleSelect} autofocus={false} />
         ) : null}

         <UserList editable={editable} onChange={handleChangeList} value={internalValue} />
      </div>
   );
});

RoleEditor.displayName = 'RoleEditor';
