import {
   autobind,
   FieldLayout,
   IDismountedProps,
   IFieldProps,
   toasts,
   toggleSetItem,
   withDismounted,
} from '@yandex-infracloud-ui/libs';
import { Button, TextInput } from 'lego-on-react';
import { array, bool } from 'prop-types';
import * as React from 'react';
import { ComponentType, SyntheticEvent } from 'react';
import { takeUntil } from 'rxjs/operators';

import { VLANEditableList } from '../../actions/host_actions/components/VLANEditableList';

import { IHost, IVlanValue } from '../../models';
import { projectApi } from '../../services';
import styles from './VLANsSelectField.module.css';

interface IState {
   added: Set<number>;
   isLoading: boolean;
   native?: number;
   projectVlans: Set<number>;
   text: string;
}

type Props = IFieldProps<IVlanValue, IHost[]> & {
   canSelect?: boolean;
};

// TODO отрабоать редактирование VLAN для нескольких хостов из разных проектов
class VLANsSelectField extends React.PureComponent<Props & IDismountedProps, IState> {
   public static propTypes = {
      canSelect: bool.isRequired,
      context: array.isRequired,
   };

   public state: IState = {
      added: new Set(),
      isLoading: false,
      projectVlans: new Set(),
      text: '',
   };

   public render() {
      return (
         <FieldLayout config={this.props.config} help={this.props.help} error={this.props.error}>
            <div className={styles.inputGroup}>
               <TextInput
                  theme={'normal'}
                  size={'s'}
                  type={'number'}
                  placeholder={'Type VLAN...'}
                  cls={styles.input}
                  text={this.state.text}
                  onChange={this._onTextChange}
                  onKeyDown={this._onEnter}
                  suggest={true}
                  suggestUrl={''}
                  showListOnFocus={true}
                  dataprovider={this._dataProvider}
               />

               <Button
                  theme={'normal'}
                  size={'s'}
                  text={'Add'}
                  onClick={this._onAdd}
                  name={'add'}
                  disabled={!this._canAdd()}
               />
            </div>

            {this.state.added.size > 0 ? (
               <VLANEditableList
                  items={this.state.added}
                  onItemsChange={this._onAddedChange}
                  value={this.state.native}
                  onChange={this._onNativeChange}
                  canSelect={this.props.canSelect!}
               />
            ) : this.props.canSelect ? (
               <p>Project and host's extra VLANs will be used</p>
            ) : null}
         </FieldLayout>
      );
   }

   public componentDidMount(): void {
      const hosts = this.props.context!;

      this.setState({ isLoading: true }, () => {
         projectApi
            .getById(hosts[0].project, ['owned_vlans'])
            .pipe(takeUntil(this.props.dismounted!))
            .subscribe(
               project =>
                  this.setState({
                     isLoading: false,
                     projectVlans: project.owned_vlans!,
                  }),
               resp => {
                  toasts.apiError('Project VLANs loading', resp);
                  this.setState({ isLoading: false });
               },
            );
      });
   }

   private _canAdd() {
      return this.state.projectVlans.has(parseInt(this.state.text, 10));
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _dataProvider(url: string, text: string, cb: (result: string[]) => void): void {
      const items = Array.from(this.state.projectVlans)
         .filter(vlan => vlan.toString().includes(text))
         .filter(vlan => !this.state.added.has(vlan))
         .map(i => i.toString());

      cb(items);
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onAdd(e: SyntheticEvent) {
      const item = parseInt(this.state.text, 10);

      this.setState(
         state => {
            // При добавлении первого элемента делаем его сразу выбранным
            const native = state.added.size === 0 ? item : state.native;

            const added = toggleSetItem(state.added, item);

            return {
               added,
               native,
               text: '',
            };
         },
         () => this._updateValue(e),
      );
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onAddedChange(e: SyntheticEvent, added: Set<number>): void {
      this.setState(
         state => ({
            added,
            native: added.size === 0 ? undefined : state.native,
         }),
         () => this._updateValue(e),
      );
   }

   @autobind
   private _onEnter(e: React.KeyboardEvent): void {
      if (e.key === 'Enter' && this._canAdd()) {
         e.preventDefault();

         this._onAdd(e);
      }
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onNativeChange(e: SyntheticEvent, native: number): void {
      this.setState({ native }, () => this._updateValue(e));
   }

   @autobind
   private _onTextChange(text: string): void {
      this.setState({ text });
   }

   private _updateValue(e: React.SyntheticEvent) {
      this.props.onChange(e, {
         items: this.state.added,
         selected: this.state.native,
      });
   }
}

const VLANsSelectFieldEnhanced = withDismounted(VLANsSelectField);

export function createVLANsSelectField({ canSelect }: { canSelect: boolean }) {
   const result: ComponentType<Props & IDismountedProps> = (props: Props) => {
      return <VLANsSelectFieldEnhanced canSelect={canSelect} {...props} />;
   };

   result.displayName = `VLANsSelectFieldEnhanced(editable=${canSelect})`;

   return result;
}
