import React, { useState, useCallback } from 'react';
import styles from './KeyValueField.module.css';
import {
   DeclarativeFields,
   ExtendedFieldConfig,
   FieldLayout2,
   InputField2,
   LazyDropdown,
   isEmpty,
   useExtendedField,
} from '@yandex-infracloud-ui/libs-next';
import { Form } from 'formik';
import { TextInput } from '@yandex-data-ui/common';
import { AddIcon } from 'design/icons';

export const KeyValueField = React.memo((props: ExtendedFieldConfig) => {
   const getInitialFields = () => {
      if (!props.value) {
         return [];
      }

      return Object.keys(props.value).map(key => ({
         as: InputField2,
         label: key,
         name: `${String(props.name)}.${key}`,
      }));
   };

   // hooks
   const [opened, setOpened] = useState(false);
   const [value, setValue] = useState('');
   const [fields, setFields] = useState(getInitialFields());

   const { readonly, onChange } = useExtendedField(props);

   // handlers
   const onChangeOpened = useCallback((v: boolean) => setOpened(v), []);

   const close = useCallback(() => setOpened(false), []);

   const updateFields = useCallback(
      (label: string) => {
         // field with selected id already exists, skip
         if (fields.map(el => el.name.toString()).includes(label)) {
            return;
         }

         setFields([
            ...fields,
            {
               as: InputField2,
               label,
               name: `${String(props.name)}.${label}`,
            },
         ]);

         onChange({ ...props.value, [label]: '' });
      },
      [fields, props, onChange],
   );

   const onInputChange = useCallback((v: string | number) => setValue(String(v)), []);

   const onKeyDown = useCallback(
      (e: React.KeyboardEvent<HTMLElement>) => {
         switch (e.key) {
            case 'Enter':
               if (!isEmpty(value)) {
                  setValue('');
                  updateFields(value);
                  close();
               }
               break;
         }
      },
      [close, updateFields, value],
   );

   // render
   if (readonly && fields.length === 0) {
      return null;
   }

   const switcher = (
      <span className={styles.addLabel}>
         Add field
         <AddIcon />
      </span>
   );

   return (
      <>
         {fields.length > 0 ? (
            <div className={styles.fullWidth}>
               <h4>Labels</h4>

               <Form>
                  <DeclarativeFields configs={fields} />
               </Form>
            </div>
         ) : null}

         <FieldLayout2 {...props}>
            <LazyDropdown switcher={switcher} opened={opened} onChangeOpened={onChangeOpened}>
               <div className={styles.popup}>
                  <TextInput
                     size={'s'}
                     value={value}
                     onUpdate={onInputChange}
                     onKeyDown={onKeyDown}
                     placeholder={props.placeholder}
                     hasClear={true}
                  />
               </div>
            </LazyDropdown>
         </FieldLayout2>
      </>
   );
});

KeyValueField.displayName = 'KeyValueField';
