import { Suggest } from '@yandex-data-ui/common';
import {
   classNames,
   FieldLayout2,
   getStrategy,
   ExtendedFieldConfig,
   ISuggestBaseEntity,
   SuggestLayer,
   useExtendedField,
   arrayToSet,
} from '@yandex-infracloud-ui/libs-next';
import { xor } from 'lodash';
import { default as React, useCallback, useState, SyntheticEvent } from 'react';

import { OwnersList } from '../../shared';

import styles from './OwnersField.module.css';

export function OwnersField(props: ExtendedFieldConfig<any, string[], { layer: SuggestLayer }>) {
   const { disabled, field, onChange, readonly } = useExtendedField(props);
   const strategy = getStrategy(props.controlProps!.layer);

   const [value, setValue] = useState('');

   const handleChange = useCallback(
      (v: string) => {
         onChange(xor(field.value, [v]));
         setValue('');
      },
      [field.value, onChange],
   );

   const onSelect = useCallback(
      (item: ISuggestBaseEntity | null) => {
         if (item === null) {
            return;
         }

         const loginOrSlug = strategy.getQueryFromEntity(item);

         const id = item.layer === SuggestLayer.Groups ? `@${loginOrSlug}` : loginOrSlug;

         onChange(xor(field.value, [id]));
         setValue('');
      },
      [field.value, onChange, strategy],
   );

   const handleGetItems = useCallback(
      async (query): Promise<ISuggestBaseEntity[] | any> => {
         if (!query) return [];

         return strategy
            .load(query)
            .toPromise()
            .then((resp: ISuggestBaseEntity[]) => resp.slice(0, 10));
      },
      [strategy],
   );

   return (
      <FieldLayout2 {...props}>
         {readonly ? null : (
            <Suggest
               getItems={handleGetItems}
               text={value}
               getItemsOnMount={true}
               onUpdate={setValue}
               onInputEnterKeyDown={handleChange}
               onItemClick={onSelect}
               applicableInputValue={true}
               placeholder={props.placeholder}
               renderItem={item => strategy.renderItem(item as any, value)}
               debounce={500}
               disabled={disabled}
               syncPopupOnResize={false}
            />
         )}

         <div>
            <OwnersList
               cls={classNames({
                  [styles.userListMarginTop]: !readonly,
               })}
               forceAvatars={true}
               value={arrayToSet(field.value)}
               editable={!disabled && !readonly}
               onChange={(e: SyntheticEvent, v: Set<string>) => {
                  onChange(Array.from(v));
               }}
            />
         </div>
      </FieldLayout2>
   );
}
