import { Suggest } from '@yandex-data-ui/common';
import {
   classNames,
   ExtendedFieldConfig,
   FieldLayout2,
   useExtendedField,
   EMPTY_VALUE,
   hasIncluded,
} from '@yandex-infracloud-ui/libs';
import React, { useCallback, useMemo } from 'react';
import { ticketApi } from 'services';
import { SuggestSelectItem, useConstants } from 'shared';

import classes from './fields.module.css';

interface ISuggestItem {
   id?: string;
   display?: string;
   key: string;
   self?: string;
   group?: boolean;
   disabled?: boolean;
   delimiter?: boolean;
}

const walleKnownTitle = {
   key: 'Wall-E known queue',
   group: true,
   disabled: true,
};
const apiTitle = {
   key: 'Other',
   group: true,
   disabled: true,
};

export const QueueConstantField = (props: ExtendedFieldConfig<any, string | null, {}>) => {
   const { disabled, field, onBlur, onChange, readonly, showError } = useExtendedField(props);
   const { constants } = useConstants();
   const walleKnownQueue = useMemo(() => constants?.known_queues_for_tickets.map(e => ({ key: e })), [constants]);

   const handleChange = useCallback(
      (v: string) => {
         onChange(v || null);
      },
      [onChange],
   );

   const handleClick = useCallback(
      (v: ISuggestItem) => {
         onChange(v.key || null);
      },
      [onChange],
   );

   const getSuggestions = (query: string) => {
      if (!query) return [];

      return ticketApi
         .getQueue(query)
         .toPromise()
         .then((resp: ISuggestItem[]) => resp.slice(0, 10));
   };

   const handleGetItems = useCallback(
      async (query): Promise<ISuggestItem[]> => {
         let result: ISuggestItem[] = [];

         const filteredWalleKnownQueue = walleKnownQueue!.filter(item => hasIncluded(query, item.key));
         const apiSuggestions = await getSuggestions(query);

         if (filteredWalleKnownQueue.length) {
            result = result.concat(walleKnownTitle, filteredWalleKnownQueue);
         }

         if (apiSuggestions.length) {
            result = result.concat(apiTitle, apiSuggestions);
         }

         return result;
      },
      [walleKnownQueue],
   );

   return (
      <FieldLayout2 {...props}>
         {readonly ? (
            <div className={classes.readonlyValue}>{field.value || EMPTY_VALUE}</div>
         ) : (
            <Suggest
               getItems={handleGetItems}
               text={field.value || ''}
               onBlur={onBlur}
               className={classNames(classes.select, { [classes.error]: showError })}
               onUpdate={handleChange}
               onItemClick={handleClick}
               onInputEnterKeyDown={handleChange}
               applicableInputValue={true}
               getItemsOnMount={true}
               showItemsOnNoText={true}
               disabled={disabled || readonly}
               placeholder={'Enter queue for ticket'}
               renderItem={item => {
                  if (item.delimiter) {
                     return <div style={{ borderBottom: '1px solid var(--yc-color-line-solid)' }} />;
                  }

                  return <SuggestSelectItem title={item.key} group={item.group} />;
               }}
               debounce={500}
            />
         )}
      </FieldLayout2>
   );
};
