import { TextInput, Button } from '@yandex-data-ui/common';
import {
   classNames,
   FieldLayout2,
   ExtendedFieldConfig,
   useExtendedField,
   EMPTY_VALUE,
} from '@yandex-infracloud-ui/libs-next';

import { xor } from 'lodash';
import { default as React, useCallback, useState, SyntheticEvent, KeyboardEvent, ChangeEvent } from 'react';

import styles from './SetListInput.module.css';

import { CloseIcon } from 'design/icons';

interface IProps {
   cls?: string;
   editable: boolean;
   forceAvatars?: boolean;
   value: string[];

   onChange?(e: SyntheticEvent, v: string[]): void;
}

const SetList = React.memo(({ cls = '', editable, onChange, value }: IProps) => {
   if (editable && !onChange) {
      throw new Error('UserList need onChange if editable');
   }

   // handlers
   const remove = (item: string) => (e: SyntheticEvent) => {
      onChange!(e, xor(value, [item]));
   };

   // render
   if (!editable && value.length === 0) {
      return <div>{EMPTY_VALUE}</div>;
   }

   const itemClass = classNames(styles.item, {
      [styles.editable]: editable,
   });

   const set = value.map(item => (
      <div key={item} className={itemClass}>
         {item}

         {editable ? (
            <Button view={'clear'} size={'s'} className={styles.deleteButton} onClick={remove(item)}>
               <CloseIcon size={'lg'} />
            </Button>
         ) : null}
      </div>
   ));

   return <div className={classNames(styles.userList, cls)}>{set}</div>;
});

export const SetListInput = (props: ExtendedFieldConfig<any, string[], {}>) => {
   const { disabled, field, onChange, readonly } = useExtendedField(props);

   const [value, setValue] = useState('');

   const handleChange = useCallback(
      (v: string) => {
         onChange(xor(field.value, [v]));
      },
      [field.value, onChange],
   );

   const handlEnterPress = useCallback(
      (e: KeyboardEvent<HTMLInputElement | HTMLTextAreaElement>) => {
         if (e.keyCode === 13) {
            handleChange(e.currentTarget.value);
            setValue('');
         }
      },
      [handleChange],
   );

   const onChangeHandler = useCallback((e: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
      setValue(e.currentTarget.value);
   }, []);

   return (
      <FieldLayout2 {...props}>
         {readonly ? null : (
            <TextInput
               name={field.name}
               onKeyDown={handlEnterPress}
               value={value}
               type={'text'}
               onChange={onChangeHandler}
               placeholder={props.placeholder}
               disabled={disabled}
            />
         )}

         <div>
            <SetList
               cls={styles.setList}
               forceAvatars={true}
               value={field.value || []}
               editable={!disabled && !readonly}
               onChange={(e: SyntheticEvent, v: string[]) => {
                  onChange(v);
               }}
            />
         </div>
      </FieldLayout2>
   );
};
