import { MINUTE_IN_SEC, WEEK_IN_SEC } from '@yandex-infracloud-ui/libs';
import { addSeconds, differenceInSeconds } from 'date-fns';
import { TimeoutMode } from '../../models';

import { limitDate, limitNumber } from '../../utils/toLibs';

export const timerBounds = {
   max: WEEK_IN_SEC,
   min: 10 * MINUTE_IN_SEC,
};

//region Actions
export enum ActionType {
   SetMode = 'SetMode',
   SetDate = 'SetDate',
   SetTimer = 'SetTimer',
   Update = 'Update',
}

export const setMode = (value: TimeoutMode) => ({
   type: ActionType.SetMode as ActionType.SetMode,
   value,
});

export const setDate = (value: Date) => ({
   type: ActionType.SetDate as ActionType.SetDate,
   value,
});

export const setTimer = (value: number) => ({
   type: ActionType.SetTimer as ActionType.SetTimer,
   value,
});

export const update = () => ({
   type: ActionType.Update as ActionType.Update,
});

type Action =
   | ReturnType<typeof setMode>
   | ReturnType<typeof setDate>
   | ReturnType<typeof setTimer>
   | ReturnType<typeof update>;
//endregion

//region State
export const initialState = {
   date: null as Date | null,
   mode: TimeoutMode.Date,
   /**
    * Хак, чтобы отслеживать окончательное изменение значения (после пересчёта лимитов)
    */
   synchronizedCount: 0,
   timer: 600,
};

type State = typeof initialState;

export const initState = (state: State): State => ({
   ...state,
   timer: differenceInSeconds(state.date!, new Date()),
});

//endregion

export function reducer(state: State, action: Action): State {
   const now = new Date();

   switch (action.type) {
      case ActionType.SetDate: {
         return {
            ...state,
            date: action.value,
            timer: differenceInSeconds(action.value, now),
         };
      }

      case ActionType.SetMode: {
         return reducer({ ...state, mode: action.value }, update());
      }

      case ActionType.SetTimer: {
         return {
            ...state,
            date: addSeconds(now, action.value),
            timer: action.value,
         };
      }

      case ActionType.Update: {
         if (state.mode === TimeoutMode.Date) {
            const minDate = addSeconds(now, timerBounds.min);
            const maxDate = addSeconds(now, timerBounds.max);
            const date = limitDate(state.date!, minDate, maxDate);

            return {
               ...state,
               date,
               synchronizedCount: state.synchronizedCount + 1,
               timer: differenceInSeconds(date, now),
            };
         } else {
            const timer = limitNumber(state.timer, timerBounds.min, timerBounds.max);

            return {
               ...state,
               date: addSeconds(now, timer),
               synchronizedCount: state.synchronizedCount + 1,
               timer,
            };
         }
      }
   }
}
