import {
   classNames,
   DateTimePicker,
   DurationInput,
   ExtendedFieldConfig,
   FieldLayout2,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import { RadioButton } from '@yandex-data-ui/common';
import React, { SyntheticEvent, useCallback, useEffect, useReducer, useRef } from 'react';
import { TimeoutMode, TimeoutValue } from '../../models';

import commonClasses from './fields.module.css';

import classes from './TimeoutField.module.css';
import { initialState, initState, reducer, setDate, setMode, setTimer, update } from './TimeoutField.state';

export const TimeoutField = (props: ExtendedFieldConfig<any, TimeoutValue, {}>) => {
   const { disabled, field, onBlur, onChange, readonly, showError } = useExtendedField(props);

   const [{ mode, date, synchronizedCount, timer }, dispatch] = useReducer(
      reducer,
      {
         ...initialState,
         ...field.value,
      },
      initState,
   );

   // Чтобы не дать сработать эффекту с onChange
   const timerRef = useRef<TimeoutValue>({ date, mode, timer });

   useEffect(() => {
      timerRef.current = { date, mode, timer };
   }, [date, mode, timer]);

   useEffect(() => {
      if (synchronizedCount > 0) {
         onChange(timerRef.current);
      }
   }, [onChange, mode, synchronizedCount]);

   const handleModeChange = useCallback(
      (v: string) => {
         dispatch(setMode(v as TimeoutMode));
         onBlur();
      },
      [onBlur],
   );

   const handleDateChange = useCallback((e: SyntheticEvent, v: Date) => dispatch(setDate(v)), []);

   const handleTimerChange = useCallback((v: number) => dispatch(setTimer(v)), []);

   const forceUpdate = useCallback(() => {
      dispatch(update());
      onBlur();
   }, [onBlur]);

   return (
      <FieldLayout2 {...props}>
         <div className={classNames(classes.wrapper, { [commonClasses.error]: showError })}>
            <RadioButton
               className={classes.radio}
               onUpdate={handleModeChange}
               disabled={disabled || readonly}
               options={[
                  { content: 'Date', value: TimeoutMode.Date },
                  { content: 'Timer', value: TimeoutMode.Timer },
               ]}
               value={mode}
               name={'apiSwitcher'}
            />
            <div className={classes.value}>
               {mode === TimeoutMode.Timer ? (
                  <div className={classes.timer}>
                     <DurationInput
                        name={field.name}
                        value={timer}
                        disabled={disabled || readonly}
                        onChange={handleTimerChange}
                        onBlur={forceUpdate}
                     />
                  </div>
               ) : (
                  <DateTimePicker
                     showTimeString={true}
                     withTime={true}
                     name={field.name}
                     value={date}
                     onChange={handleDateChange}
                     onBlur={forceUpdate}
                     disabled={disabled || readonly}
                  />
               )}
            </div>
         </div>
      </FieldLayout2>
   );
};

TimeoutField.displayName = 'TimeoutField';
