import {
   autobind,
   BottomFooter,
   copyToClipboard,
   formatNumber,
   Loader,
   plural,
   styleHelpers,
} from '@yandex-infracloud-ui/libs';
import { CopyIcon } from 'design/icons';
import { Button } from 'lego-on-react';
import * as React from 'react';
import { ReactNode, SyntheticEvent } from 'react';
import { HostActionButtons } from '../../actions/host_actions';
import { IHostFilters, IHostSort } from '../../models';
import { SidebarDirection, SidebarToggleButton } from '../../shared';
import { HostFilters } from './components/HostFilters/HostFilters';
import { HostList } from './components/HostList/HostList';

import classes from './HostListScreen.module.css';
import { IHostListState, IHostStateMethods } from './HostListState';
import { SelectMode } from './models';

type Props = IHostListState & IHostStateMethods;

export class HostListScreen extends React.PureComponent<Props> {
   public render() {
      return (
         <>
            <HostFilters
               metaStatusButtons={this.props.metaStatusButtons}
               constants={this.props.constants}
               locations={this.props.locations}
               loadLocations={this.props.loadLocations}
               value={this.props.filters}
               onChange={this._onFiltersUpdate}
            />

            <HostList
               boxes={this.props.boxes}
               constants={this.props.constants}
               items={this.props.items}
               total={this.props.total}
               expanded={this.props.expanded}
               expand={this.props.toggleExpand}
               selected={this.props.selected}
               select={this.props.toggleSelected}
               onActionFinished={this.props.onActionFinished}
               sortBy={this.props.sortBy}
               onSortChange={this._onSortUpdate}
            />

            <Loader cls={classes.loader} visible={this.props.isLoading} text={'Host list loading'} />

            {this.props.items.length === 0 && !this.props.isLoading ? (
               <div className={styleHelpers.center}>No data</div>
            ) : null}

            <div className={classes.tableFooter}>
               <SidebarToggleButton className={classes.sidebarToggleButton} direction={SidebarDirection.Expand} />

               <div className={classes.pagination}>
                  {this.props.hasMore && !this.props.isLoading ? (
                     <div className={classes.loadMore}>
                        <Button theme={'action'} size={'s'} onClick={this._onNextPageClick} text={'Load more'} />
                     </div>
                  ) : null}
               </div>
            </div>

            {this.props.selected.size > 0 ? (
               <BottomFooter>
                  <HostActionButtons
                     leftText={this._renderSelectedCount()}
                     currentContext={this.props.selectedHosts}
                     disabled={!this.props.canCallHostAction}
                     getSelectedHosts={this.props.getSelectedHosts}
                     compact={false}
                     onActionFinished={this.props.onActionFinished}
                  />
               </BottomFooter>
            ) : null}
         </>
      );
   }

   @autobind
   private _onCopy() {
      const hosts = this.props.selectedHosts.map(h => h.name).join(' ');
      copyToClipboard(hosts);
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onFiltersUpdate(e: SyntheticEvent, filters: IHostFilters) {
      this.props.setState({ filters, nextCursor: 0 }, () =>
         this.props.loadHostsDebounced({
            loadStat: true,
            reason: 'filters',
            updateUrl: true,
         }),
      );
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onSortUpdate(sortBy: IHostSort[]) {
      this.props.setState({ sortBy }, () =>
         this.props.loadHostsDebounced({
            loadStat: true,
            reason: 'sorting',
            updateUrl: true,
         }),
      );
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _onNextPageClick() {
      this.props.loadHostsDebounced({
         loadStat: false,
         reason: 'page',
         updateUrl: true,
      });
   }

   private _renderSelectedCount(): ReactNode {
      const selectedCount = this.props.selected.size;

      if (selectedCount === 0) {
         return null;
      }

      const countEl = (
         <>
            <CopyIcon onClick={this._onCopy} className={classes.copyIcon} />

            <span className={classes.selected}>
               {plural(selectedCount, 'One host selected', `${formatNumber(selectedCount)} hosts selected`)}
            </span>
         </>
      );

      if (!this.props.isAllSelected()) {
         return countEl;
      }

      const selectAllButton =
         this.props.selectMode === SelectMode.Page ? (
            <>
               <Button
                  theme={'normal'}
                  size={'s'}
                  text={'Select on all pages'}
                  disabled={this.props.areHostsLoading}
                  onClick={this.props.selectAllPages}
               />

               <Loader text={'Hosts are loading'} visible={this.props.areHostsLoading} inline={true} />
            </>
         ) : (
            <Button theme={'normal'} size={'s'} text={'Select only on the page'} onClick={this.props.selectPage} />
         );

      return (
         <>
            {countEl}
            {selectAllButton}
         </>
      );
   }
}
