import {
   autobind,
   debounce,
   fromQuery,
   getSetDifference,
   IDismountedProps,
   IListOption,
   IResult,
   isEmpty,
   isEmptyOrDefault,
   isEqual,
   ITreeOption,
   omitUndefinedProperties,
   setToQueryValue,
   toasts,
   toggleSetItem,
   toQuery,
   withDismounted,
} from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { connect } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { Observable, of, Subscription } from 'rxjs';
import { finalize, map, take, takeUntil } from 'rxjs/operators';

import { HostAction } from '../../actions/host_actions';
import {
   CURSOR_PAGINATION_LIMIT,
   IApiError,
   IBoxes,
   IConstants,
   IHost,
   IHostFilters,
   IHostSort,
   ILocation,
   MetaStatusButtons,
   ProjectType,
   SLOW_UPDATE_INTERVAL,
} from '../../models';
import { auth, config, dictApi, hostApi } from '../../services';
import { IdSet, idSetToList } from '../../state/commonModels';
import { projectsSlice } from '../../state/projects';
import { RootState } from '../../state/store';

import { HostListScreen } from './HostListScreen';
import {
   ForcedHostFilters,
   getDefaultHostFilters,
   hostFiltersToUrlParams,
   IHostUrlParams,
   SelectMode,
   urlParamsToHostFilters,
} from './models';

function location2Option(location: ILocation): ITreeOption {
   const option: ITreeOption = {
      name: location.name,
      value: location.path,
   };

   if (location.nodes && location.nodes.length > 0) {
      option.children = location.nodes.map(location2Option);
   }

   return option;
}

export interface IHostListState {
   areHostsLoading: boolean;
   boxes: IBoxes | null;
   canCallHostAction: boolean;
   constants: IConstants | null;
   expanded: Set<string>;
   filters: IHostFilters;
   hasMore: boolean;
   isLoading: boolean;
   items: IHost[];
   locations: IListOption[];
   metaStatusButtons: MetaStatusButtons;
   nextCursor: number;
   selectMode: SelectMode;
   selected: Set<string>;
   // Для передачи контекста в HostActionButtons
   selectedHosts: IHost[];
   total: number;
   sortBy: IHostSort[];
}

interface ILoadOptions {
   /**
    * Нужно ли обновить краткую статистику в MetaStatusFilter
    *
    * (число хостов в разных состояниях и их процент об общего числа)
    */
   loadStat: boolean;

   /**
    * Причина загрузки
    */
   reason?: 'initial' | 'filters' | 'page' | 'sorting';

   /**
    * Нужно ли обновлять URL
    */
   updateUrl: boolean;
}

export interface IHostStateMethods {
   getSelectedHosts(): Observable<IHost[]>;

   isAllSelected(): boolean;

   loadLocations(): void;

   loadHosts(options: ILoadOptions): void;

   loadHostsDebounced(options: ILoadOptions): void;

   onActionFinished(action: HostAction, results: IResult[]): void;

   selectPage(): void;

   selectAllPages(): void;

   setState<K extends keyof IHostListState, P extends Props, S extends IHostListState>(
      state: StateHandler<S, P, K>,
      callback?: () => void,
   ): void;

   toggleExpand(uuid: string): void;

   toggleSelected(uuids: Set<string>): void;
}

const mapStateToProps = (state: RootState) => ({
   selectedProjects: state.projects.selectedIds,
   userProjects: state.globals.user && state.globals.user.source === 'api' ? state.globals.user.projects : undefined,
});

const mapDispatchToProps = { selectProjects: projectsSlice.actions.select };

type Props = RouteComponentProps &
   ReturnType<typeof mapStateToProps> &
   typeof mapDispatchToProps & {
      forceFilters?: ForcedHostFilters;
      forceIds?: Set<number>;
      useOwnershipFilter?: boolean;

      onChangeTotal?(total: number): void;
   };

type StateHandler<S extends IHostListState, P extends Props, K extends keyof IHostListState> =
   | ((prevState: Readonly<S>, props: Readonly<P>) => Pick<S, K> | S | null)
   | Pick<S, K>
   | S
   | null;

class HostListState extends React.PureComponent<Props & IDismountedProps, IHostListState> {
   public static defaultProps = {
      useOwnershipFilter: false,
   };

   private static readonly _initialState: Readonly<IHostListState> = {
      areHostsLoading: false,
      boxes: null,
      canCallHostAction: false,
      constants: null,
      expanded: new Set(),
      filters: getDefaultHostFilters(),
      hasMore: false,
      isLoading: true,
      items: [],
      locations: [],
      metaStatusButtons: new MetaStatusButtons(),
      nextCursor: 0,
      selectMode: SelectMode.Page,
      selected: new Set(),
      selectedHosts: [],
      total: 0,
      sortBy: [],
   };

   // Для прерывания загрузки хостов при смене фильтров
   private _loadHostSubscription: Subscription | null = null;

   // Для прерывания загрузки статистики хостов при смене фильтров
   private _loadHostsStatSubscription: Subscription | null = null;

   private _updateTimerId: number | null = null;

   private _wasLoaded = false;

   constructor(props: Props) {
      super(props);

      this.state = HostListState._initialState;
   }

   /**
    * @deprecated (я не знаю, что курил, когда это писал)
    */
   @autobind
   public setState<K extends keyof IHostListState, S extends IHostListState>(
      state: Pick<S, K> | S | null,
      callback?: () => void,
   ): void {
      // @ts-ignore // TODO разобраться
      super.setState(state, () => {
         if (callback) {
            callback();
         }

         // HOOK
         const newProject = (state as any)?.filters?.project; // FIXME
         if (!this._hasForcedProject() && newProject) {
            this.props.selectProjects(Array.from(this.state.filters.project));
         }
      });
   }

   public componentDidMount() {
      this._initState().then(() => {
         if (!this.props.useOwnershipFilter!) {
            this._firstLoad();

            return;
         }
         // will be loaded after user loading in componentDidUpdate
      });
   }

   public componentWillUnmount(): void {
      this._unscheduleUpdating();
   }

   public componentDidUpdate(prevProps: Props): void {
      if (prevProps.location.search !== this.props.location.search) {
         this._updateFiltersFromUrl();
      }

      if (this.props.useOwnershipFilter && this.props.userProjects !== undefined) {
         this._firstLoad();
      }

      if (!isEqual(prevProps.selectedProjects, this.props.selectedProjects)) {
         this._onProjectsChange(this.props.selectedProjects);
      }
   }

   public render(): React.ReactNode {
      return (
         <HostListScreen
            {...this.state}
            getSelectedHosts={this._getSelectedHosts}
            isAllSelected={this._isAllSelected}
            loadHosts={this._loadHosts}
            loadHostsDebounced={this._loadHostsDebounced}
            loadLocations={this._loadLocations}
            onActionFinished={this._onActionFinished}
            selectAllPages={this._selectAllPages}
            selectPage={this._selectPage}
            setState={this.setState as any}
            toggleExpand={this._toggleExpand}
            toggleSelected={this._toggleSelected}
         />
      );
   }

   private _firstLoad() {
      if (this._wasLoaded) {
         return;
      }

      this._wasLoaded = true;
      this._loadHosts({ loadStat: true, reason: 'initial', updateUrl: false });
   }

   private _getFiltersForApi(): IHostFilters {
      return {
         ...this.state.filters,
         ...this.props.forceFilters,
         ids: this.props.forceIds,
      };
   }
   private _getSortingForApi(): string {
      const { sortBy } = this.state;

      return sortBy.reduce((acc, el) => {
         if (!el.direction) return acc;
         if (acc.length > 0) acc += ',';

         switch (el.name) {
            case 'links':
               return (acc += `ticket:${el.direction},scenario_id:${el.direction}`);

            default:
               return (acc += `${el.name}:${el.direction}`);
         }
      }, '');
   }

   @autobind
   private _getSelectedHosts(): Observable<IHost[]> {
      switch (this.state.selectMode) {
         case SelectMode.AllPages:
            return hostApi.getAll(
               this._getFiltersForApi(),
               this.props.useOwnershipFilter!,
               this.props.userProjects,
               this.state.total,
            );

         case SelectMode.Page:
            return of(this.state.selectedHosts);
      }
   }

   private _hasForcedProject() {
      return this.props.forceFilters && this.props.forceFilters.project && this.props.forceFilters.project.size > 0;
   }

   /**
    * Восстановление состояния из URL, загрузка констант
    */
   @autobind
   private _initState(): Promise<void> {
      const urlParams: IHostUrlParams = fromQuery(this.props.location.search);

      const filters: IHostFilters = urlParamsToHostFilters(urlParams);

      const newState = omitUndefinedProperties<IHostListState>({
         filters: isEmptyOrDefault(filters, { status: 'all' }) ? undefined : filters,
      });

      if (!this._hasForcedProject()) {
         if (newState.filters && this.state.filters.project !== newState.filters.project) {
            this.props.selectProjects(Array.from(newState.filters.project));
         }
      }

      this._loadConstants();
      this._loadBoxes();

      if (isEmptyOrDefault(newState, HostListState._initialState)) {
         return Promise.resolve();
      }

      return new Promise(resolve => this.setState(newState as any, resolve));
   }

   @autobind
   private _isAllSelected(): boolean {
      switch (this.state.selectMode) {
         case SelectMode.AllPages:
            return this.state.selected.size > 0 && this.state.selected.size === this.state.total;

         case SelectMode.Page:
            return this.state.selected.size > 0 && this.state.selected.size === this.state.items.length;
      }
   }

   private _loadConstants() {
      dictApi
         .getConstants()
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(constants => this.setState({ constants }), toasts.handleApiError('Constants load'));
   }

   private _loadBoxes() {
      dictApi
         .getBoxes()
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(boxes => this.setState({ boxes }));
   }

   /**
    * Перезагружает список хостов с актуальными фильтрами и настройками страницы
    *
    * @param loadStat {boolean} Нужно ли обновить краткую статистику в MetaStatusFilter
    * (число хостов в разных состояниях и их процент об общего числа)
    * @param updateUrl {boolean=true} Требуется ли обновить URL страницы
    * @param reason
    */
   @autobind
   private _loadHosts({ loadStat, updateUrl, reason }: ILoadOptions) {
      if (updateUrl) {
         this._updateUrl();
      }
      this._unscheduleUpdating();

      const wasSelectedAll = this._isAllSelected();
      const append = reason === 'page';

      this.setState(
         {
            isLoading: true,
            items: append ? this.state.items : [],
            expanded: append ? this.state.expanded : new Set(),
            selected: append ? this.state.selected : new Set(),
         },
         () => {
            if (this._loadHostSubscription) {
               this._loadHostSubscription.unsubscribe();
            }

            this._loadHostSubscription = hostApi
               .getList({
                  filters: this._getFiltersForApi(),
                  sortBy: this._getSortingForApi(),
                  nextCursor: this.state.nextCursor,

                  limit: CURSOR_PAGINATION_LIMIT,
                  useOwnershipFilter: this.props.useOwnershipFilter!,
                  userProjects: this.props.userProjects,
               })
               .pipe(
                  finalize(() => (this._loadHostSubscription = null)),
                  takeUntil(this.props.dismounted!),
               )
               .subscribe(
                  resp => {
                     // Если номер страницы больше, чем их всего, то сбрасываем на первую страницу
                     // TODO
                     // if (resp.result.length === 0 && this.state.page > 1) {
                     //    this.setState({ page: 1 }, () => this._loadHosts({ loadStat, updateUrl: true }));
                     //
                     //    return;
                     // }

                     this._scheduleUpdating(resp.result);

                     const selected = new Set(wasSelectedAll ? resp.result.map(h => h.uuid) : []);

                     this.setState({
                        isLoading: false,
                        hasMore: Boolean(resp.next_cursor),
                        items: append ? [...this.state.items, ...resp.result] : resp.result,
                        nextCursor: resp.next_cursor!,
                        selected,
                        total: resp.total!,
                     });

                     if (this.props.onChangeTotal) {
                        this.props.onChangeTotal(resp.total!);
                     }
                  },
                  (resp: IApiError) => {
                     this.setState({ isLoading: false });
                     toasts.apiError('Host list loading', resp);
                  },
               );

            if (loadStat) {
               this._loadStat();
            }
         },
      );
   }

   @debounce(500)
   private _loadHostsDebounced(options: ILoadOptions) {
      return this._loadHosts(options);
   }

   @autobind
   private _loadLocations() {
      const projects = setToQueryValue(this.state.filters.project);

      dictApi
         .getLocations(projects)
         .pipe(
            map(resp => resp.result.map(location2Option)),
            takeUntil(this.props.dismounted!),
         )
         .subscribe(locations => this.setState({ locations }), toasts.handleApiError('Locations loading'));
   }

   /**
    * Обновление краткой статистики в MetaStatusFilter
    *
    * (числа хостов в разных состояниях и их процент об общего числа)
    */
   private _loadStat() {
      if (this._loadHostsStatSubscription) {
         this._loadHostsStatSubscription.unsubscribe();
      }

      this._loadHostsStatSubscription = hostApi
         .getStat(this._getFiltersForApi(), this.props.useOwnershipFilter!, this.props.userProjects)
         .pipe(
            finalize(() => (this._loadHostsStatSubscription = null)),
            takeUntil(this.props.dismounted!),
         )
         .subscribe(
            stat =>
               this.setState({
                  metaStatusButtons: this.state.metaStatusButtons.update(stat),
               }),
            toasts.handleApiError('Host short statistic loading'),
         );
   }

   @autobind
   private _onActionFinished(): void {
      this._loadHosts({ loadStat: false, updateUrl: false });
   }

   private _onProjectsChange(selectedProjects: IdSet) {
      const projects = new Set(idSetToList(selectedProjects));
      if (isEqual(projects, this.state.filters.project)) {
         this._loadLocations();

         return;
      }

      const filters: IHostFilters = { ...this.state.filters, project: projects };

      this.setState({ nextCursor: 0, filters }, () => {
         this._loadHosts({ loadStat: true, updateUrl: true });
         this._loadLocations();
      });
   }

   private _scheduleUpdating(hosts: IHost[]): void {
      this._unscheduleUpdating();

      this._updateTimerId = window.setTimeout(() => this._updateList(), SLOW_UPDATE_INTERVAL);
   }

   @autobind
   private _selectAllPages() {
      this.setState({ areHostsLoading: true });

      return hostApi
         .getAll(this._getFiltersForApi(), this.props.useOwnershipFilter!, this.props.userProjects, this.state.total)
         .pipe(take(1))
         .subscribe(hosts => {
            const selected = new Set(hosts.map(h => h.uuid));
            this.setState({
               selected,
               selectedHosts: hosts,
               selectMode: SelectMode.AllPages,
               areHostsLoading: false,
            });
         });
   }

   @autobind
   private _selectPage() {
      const hosts = this.state.items;
      const selected = new Set(hosts.map(h => h.uuid));
      this.setState({ selected, selectedHosts: hosts, selectMode: SelectMode.Page });
   }

   @autobind
   private _toggleExpand(uuid: string) {
      this.setState({ expanded: toggleSetItem(this.state.expanded, uuid) });
   }

   @autobind
   private _toggleSelected(selected: Set<string>) {
      const selectedHosts = this.state.items.filter(host => selected.has(host.uuid));

      this.setState({
         canCallHostAction: selectedHosts.every(h => auth.canRunHostAction(h)),
         selectMode: SelectMode.Page,
         selected,
         selectedHosts,
      });
   }

   private _unscheduleUpdating() {
      if (this._updateTimerId) {
         window.clearTimeout(this._updateTimerId);
         this._updateTimerId = null;
      }
   }

   private _updateFiltersFromUrl(): void {
      const urlParams: IHostUrlParams = fromQuery(this.props.location.search);
      const filters = urlParamsToHostFilters(urlParams);
      const filtersChanged = !isEqual(filters, this.state.filters);
      if (!filtersChanged) {
         return;
      }

      this.setState({ filters }, () => this._loadHosts({ loadStat: true, updateUrl: false }));
   }

   private _updateList() {
      hostApi
         .getList({
            filters: this._getFiltersForApi(),
            sortBy: this._getSortingForApi(),
            limit: this.state.items.length, // TODO уточнить поведение
            nextCursor: 0,
            useOwnershipFilter: this.props.useOwnershipFilter!,
            userProjects: this.props.userProjects,
         })
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(
            resp => {
               const existUuids = new Set(this.state.items.map(h => h.uuid));
               const newUuids = new Set(resp.result.map(h => h.uuid));
               const { added, removed } = getSetDifference(existUuids, newUuids);
               let hasUpdates = !isEmpty(added) || !isEmpty(removed);

               // императивщина, т.к грязный код (сложная мутация hasUpdates)
               const items: IHost[] = [];
               for (const host of resp.result) {
                  // Если был добавлен, просто добавляем
                  if (added.has(host.uuid)) {
                     items.push(host);
                     continue;
                  }

                  // Проверяю изменения
                  const existHost = this.state.items.find(h => h.uuid === host.uuid)!;
                  if (isEqual(host, existHost)) {
                     items.push(existHost);
                  } else {
                     hasUpdates = true;
                     items.push(host);
                  }
               }

               if (hasUpdates) {
                  const selectedHosts = items.filter(h => this.state.selected.has(h.uuid));

                  this.setState({
                     canCallHostAction: selectedHosts.every(
                        h => auth.canRunHostAction(h) && h.type !== ProjectType.SHADOW,
                     ),
                     items,
                     selectedHosts,
                     total: resp.total!,
                  });

                  if (this.props.onChangeTotal) {
                     this.props.onChangeTotal(resp.total!);
                  }
               }

               this._scheduleUpdating(resp.result);
            },
            resp => {
               toasts.apiError('Host list updating', resp);
               this._scheduleUpdating(this.state.items);
            },
         );
   }

   @autobind
   private _updateUrl(): void {
      const params = hostFiltersToUrlParams(this.state.filters);

      if (config.profiling) {
         // tslint:disable-next-line:no-string-literal
         params['react_perf'] = null;
      }

      const search = toQuery(params);
      if (search !== this.props.location.search) {
         this.props.history.push({ search });
      }
   }
}

/**
 * @deprecated
 */
export const HostListStateEnhanced = connect(mapStateToProps, mapDispatchToProps)(withDismounted(HostListState));
