import { isEqual, ITreeOption } from '@yandex-infracloud-ui/libs';
import { Button } from 'lego-on-react';
import * as React from 'react';
import { SyntheticEvent, useEffect, useState } from 'react';

import { IConstants, IHostFilters, MetaStatusButtons } from '../../../../models';

import { getDefaultHostFilters } from '../../models';
import { FqdnField } from './fields/FqdnField';
import { LocationsField } from './fields/LocationsField';
import { MetaStatusField } from './fields/MetaStatusField';
import { OptionsField } from './fields/OptionsField';
import { TagsField } from './fields/TagsField';
import { getUsedAdditionalFilterCount, toListOption } from './helpers';
import styles from './HostFilters.module.css';

interface IProps {
   constants: IConstants | null;
   locations: ITreeOption[];
   metaStatusButtons: MetaStatusButtons;
   value: IHostFilters;

   onChange(e: SyntheticEvent | null, v: IHostFilters): void;

   loadLocations(): void;
}

export const HostFilters = ({ constants, loadLocations, locations, metaStatusButtons, onChange, value }: IProps) => {
   // hooks
   const [keepOpened, setKeepOpened] = useState(false);
   const [opened, setOpened] = useState(false);

   useEffect(() => {
      const needToOpen = getUsedAdditionalFilterCount(value) > 0;
      setOpened(needToOpen);

      if (needToOpen) {
         setKeepOpened(true);
      }
   }, [value]);

   // handlers
   const updater = <K extends keyof IHostFilters>(field: K) => (e: SyntheticEvent | null, v: IHostFilters[K]) => {
      if (isEqual(v, value[field])) {
         return;
      }

      onChange(null, { ...value, [field]: v });
   };

   const changeHandlers = {
      config: updater('config'),
      fqdn: updater('fqdn'),
      health: updater('health'),
      locations: updater('locations'),
      restrictions: updater('restrictions'),
      state: updater('state'),
      status: updater('status'),
      switch: updater('switch'),
      types: updater('type'),
   };

   const onToggleOpened = () => {
      setOpened(!opened);
      setKeepOpened(!opened);
   };

   const onReset = () => {
      const defaultFilters = getDefaultHostFilters();

      if (!isEqual(value, defaultFilters)) {
         onChange(null, defaultFilters);
      }
   };

   // render
   const healthStatuses = constants ? constants.health_statuses.map(toListOption) : [];
   const hostStates = constants ? constants.host_states.map(toListOption) : [];
   const hostStatuses = constants ? constants.host_statuses.map(toListOption) : [];
   const restrictions = constants ? constants.restrictions.map(toListOption) : [];
   const types = constants ? constants.host_types.map(toListOption) : [];

   return (
      <div className={styles.hostFilters}>
         <FqdnField value={value.fqdn} onChange={changeHandlers.fqdn} />

         <MetaStatusField
            metaStatusButtons={metaStatusButtons}
            value={value}
            onChange={onChange}
            opened={opened}
            onToggle={onToggleOpened}
         />

         {opened || keepOpened ? (
            <>
               <LocationsField
                  locations={locations}
                  loadLocations={loadLocations}
                  value={value.locations}
                  onChange={changeHandlers.locations}
               />

               <OptionsField label={'State'} options={hostStates} value={value.state} onChange={changeHandlers.state} />

               <OptionsField
                  label={'Status'}
                  options={hostStatuses}
                  value={value.status}
                  onChange={changeHandlers.status}
               />

               <OptionsField
                  label={'Health'}
                  options={healthStatuses}
                  value={value.health}
                  onChange={changeHandlers.health}
               />

               <OptionsField
                  label={'Restrictions'}
                  options={restrictions}
                  value={value.restrictions}
                  onChange={changeHandlers.restrictions}
               />

               <OptionsField label={'Types'} options={types} value={value.type} onChange={changeHandlers.types} />

               <TagsField placeholder={'Switch'} value={value.switch} onChange={changeHandlers.switch} />

               <TagsField placeholder={'Deploy configuration'} value={value.config} onChange={changeHandlers.config} />

               <div>
                  <Button theme={'normal'} size={'s'} onClick={onReset}>
                     Reset
                  </Button>
               </div>
            </>
         ) : null}
      </div>
   );
};

HostFilters.displayName = 'HostFilters';
