import { classNames, CopyableText, EMPTY_VALUE, IResult, styleHelpers } from '@yandex-infracloud-ui/libs';
import { Button, Icon } from 'lego-on-react';
import * as React from 'react';
import { Children, ReactNode, SyntheticEvent, useContext } from 'react';
import { useSelector } from 'react-redux';
import { useLocation } from 'react-router';
import { Link } from 'react-router-dom';
import { of } from 'rxjs';
import { RootState } from 'state/store';

import { HostAction, HostActionButtons } from '../../../../actions/host_actions';
import {
   EXTERNAL_LINKS,
   HostMessageSeverity,
   IConstantGroups,
   IHost,
   ROUTE_LINKS,
   IBoxes,
   ProjectType,
} from '../../../../models';
import {
   ClickOnlyUnmovedTrap,
   HostHealth,
   HostLinks,
   HostListContext,
   HostStatusMessages,
   LegoCheckbox,
   ProjectName,
   RequestHostAccessLink,
   TrackerLink,
   useCanRunHostAction,
} from '../../../../shared';

import { HostRestrictions } from '../HostRestrictions/HostRestrictions';
import styles from './HostRow.module.css';

interface IProps {
   boxes?: IBoxes | null;
   expanded: boolean;
   healthGroups: IConstantGroups | null;
   host: IHost;
   selected: boolean;

   expand(uuid: string): void;

   select(e: React.KeyboardEvent<HTMLInputElement>): void;

   onActionFinished?(action: HostAction, results: IResult[]): void;
}

export const HostRow = React.memo(
   ({ expand, expanded = false, select, selected = false, host, healthGroups, onActionFinished, boxes }: IProps) => {
      const location = useLocation();
      const hostKey = host.name || host.inv.toString();

      // hooks
      const canRunAction = useCanRunHostAction(host);

      const hostListContext = useContext(HostListContext);

      const { additionalColumns, hideProjectColumn } = hostListContext;

      const user = useSelector((s: RootState) => s.globals.user);

      const isAdmin = user ? user.isAdmin : false;

      const isShadow = host.type === ProjectType.SHADOW;

      const cells: ReactNode[] = [
         <td className={styles.checkColumn} key={'checkColumn'}>
            <LegoCheckbox checked={selected} onChange={select} />
         </td>,

         <td key={'host'} className={classNames(styles.nowrap, styles.withPadding)}>
            <span className={styles.copyText}>
               {!host.name ? 'empty FQDN, inv#' : null}
               <Link
                  to={{
                     pathname: ROUTE_LINKS.host(host.name, host.inv),
                     state: { prev: location },
                  }}
               >
                  {hostKey}
               </Link>
               <CopyableText className={styles.copyIcon} text={hostKey} />
            </span>

            {expanded && host.name ? (
               <>
                  <div className={classNames(styles.row, styles.copyText)}>
                     <div>
                        Inv: <span className={styles.id}>{host.inv}</span>
                        <CopyableText text={host.inv.toString()} className={styles.copyIcon} />
                     </div>

                     <div>
                        UUID: <span className={styles.id}>{host.uuid}</span>
                        <CopyableText text={host.uuid} className={styles.copyIcon} />
                     </div>
                  </div>
               </>
            ) : null}
            {isAdmin && <span className={styles.tierInfo}>tier {host.tier}</span>}
         </td>,

         hideProjectColumn ? null : (
            <td key={'project'} className={classNames(styles.nowrap, styles.withPadding)}>
               <Link to={{ pathname: ROUTE_LINKS.project(host.project), state: { prev: location } }}>
                  <ProjectName id={host.project} />
               </Link>
            </td>
         ),

         <td key={'type'} className={classNames(styles.nowrap, styles.withPadding)}>
            {host.type}
         </td>,

         <td key={'state'} className={styles.withPadding}>
            {host.state}
         </td>,

         <td key={'status'} className={styles.withPadding}>
            {host.status}
            {host.task ? `:${host.task.status}` : null}
         </td>,

         <td key={'host health'} className={styles.withPadding}>
            {host.health && healthGroups && !isShadow ? (
               <HostHealth groups={healthGroups} value={host.health.check_statuses} restrictions={host.restrictions} />
            ) : (
               EMPTY_VALUE
            )}
         </td>,

         <td key={'restrictions'} className={styles.withPadding}>
            <HostRestrictions value={host.restrictions} host={host.name} />
         </td>,
         <td key={'switch'} className={classNames(styles.nowrap, styles.withPadding)}>
            {host.location.switch}
            {expanded ? (
               <div>
                  <a
                     href={EXTERNAL_LINKS.rackTables(host.location.switch, host.location.port)}
                     className={classNames(styles.rackTableLink, styleHelpers.externalGrayLink)}
                     target={'_blank'}
                     rel={'noopener noreferrer'}
                  >
                     {host.location.port}
                  </a>
               </div>
            ) : null}
         </td>,

         <td key={'city'} className={styles.withPadding}>
            {host.location.city}
         </td>,

         <td key={'links'} className={classNames(styles.linksColumn, styles.withPadding)}>
            {host.ticket ? <TrackerLink ticket={host.ticket} /> : null}
            {host.scenario_id ? (
               <Link to={{ pathname: ROUTE_LINKS.scenario(host.scenario_id), state: { prev: location } }}>
                  Scenario #{host.scenario_id}
               </Link>
            ) : null}
         </td>,
      ];

      additionalColumns.forEach(column => {
         cells.splice(column.order, 0, <td key={column.title}>{column.render(host)}</td>);
      });

      // handlers
      const expandRow = (e: SyntheticEvent) => {
         const target = e.target as EventTarget & Element;

         if (target.tagName === 'TD') {
            expand(host.uuid);
         }
      };

      // render
      const firstRow = (
         <ClickOnlyUnmovedTrap
            as={'tr'}
            onClick={expandRow}
            data-e2e={'HostRow:first'}
            className={classNames(styles.hostRow, {
               [styles._expanded]: expanded,
               [styles._selected]: selected,
            })}
         >
            {cells}
         </ClickOnlyUnmovedTrap>
      );

      const columnCount = Children.count(firstRow.props.children);

      return (
         <>
            {firstRow}

            {expanded ? (
               <tr data-e2e={'HostRow:second'} className={styles._additional}>
                  <td />
                  <td colSpan={columnCount - 1}>
                     <HostStatusMessages cls={styles.statusMessage} host={host} severity={HostMessageSeverity.Error} />

                     <HostActionButtons
                        compact={true}
                        currentContext={[host]}
                        getSelectedHosts={() => of([host])}
                        disabled={!canRunAction}
                        onActionFinished={onActionFinished}
                        isShadow={isShadow}
                     />

                     <RequestHostAccessLink host={host} />

                     <HostLinks
                        cls={styles.links}
                        name={host.name}
                        inv={host.inv}
                        project={host.project}
                        boxes={boxes}
                     />

                     <Button
                        theme={'clear'}
                        size={'xs'}
                        cls={styles.closeButton}
                        iconLeft={<Icon type={'close'} />}
                        onClick={() => expand(host.uuid)}
                        text={''}
                     />
                  </td>
               </tr>
            ) : null}
         </>
      );
   },
);

HostRow.displayName = 'HostRow';
