import {
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { dnsFormValidationSchema, IApiError, IDnsAutomationCreditsForm, IDnsDomainForm } from '../../models';
import { projectApi } from '../../services';

import classes from '../../design/commonModals.module.css';
import { DnsDomainField } from './components/DnsDomainField';
import { DnsZoneIdField } from './components/DnsZoneIdField';
import { ReasonField } from '../form_fields2/ReasonField';

const initialValue: IDnsDomainForm = {
   dns_domain: '',
   yc_dns_zone_id: '',
   reason: '',
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <DnsDomainField name={'dns_domain'} />
      <DnsZoneIdField name={'yc_dns_zone_id'} />

      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<IDnsDomainForm> {
   dnsDomain: string;
   dnsZoneId?: string;
   id: string;
}

export const DnsDomainModal: React.FC<Props> = React.memo(({ dnsDomain, dnsZoneId, id, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IDnsAutomationCreditsForm>),
      [],
   );

   const initialValuePatched = useMemo(
      () => ({ ...initialValue, dns_domain: dnsDomain, yc_dns_zone_id: dnsZoneId ?? '' }),
      [dnsDomain, dnsZoneId],
   );

   const handleSubmit = useCallback(
      (value: IDnsDomainForm, helpers: FormikHelpers<IDnsDomainForm>) => {
         projectApi
            .updateDnsDomain(id, value.dns_domain, value.reason, value.yc_dns_zone_id)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(value);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} showFooter={false} title={'DNS'}>
         <Formik initialValues={initialValuePatched} onSubmit={handleSubmit} validationSchema={dnsFormValidationSchema}>
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'DNS is updating'} />

                        <Button
                           theme={'clear'}
                           size={'s'}
                           cls={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           theme={'action'}
                           size={'s'}
                           cls={classes.button}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting || !form.dirty}
                        >
                           Update
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
});

DnsDomainModal.displayName = 'DnsDomainModal';
